from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework import status
from .views import *
from APIs.models import Comment, User
from ziluck_project.hospital_config import validate_comment_api_key
import json


@api_view(["POST"])
def post_comment(request):
    """
    API endpoint to create a new comment
    Required fields: token, phone_number, content
    """
    try:
        # Validate required fields
        RequsetChecker(request.POST, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            },
            {
                "name": "content",
                "format": ".+"
            }
        ], request)

        token = str(request.POST["token"])
        content = str(request.POST["content"])

        # Validate user
        u = User.objects.filter(token=token)
        if not u.exists():
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)
        user = u[0]

        # Validate content length
        if len(content.strip()) < 1:
            return myResponse.Error("محتوای کامنت نمی‌تواند خالی باشد", 400)

        # Create comment
        comment = Comment.objects.create(
            user=user,
            content=content.strip()
        )

        # Return success response
        return myResponse.OK("کامنت با موفقیت ثبت شد", {
            "comment_id": comment.id,
            "content": comment.content,
            "created_at": str(comment.created_at),
            "user": user.identifier
        })

    except Exception as e:
        return myResponse.Error(f"خطا در ثبت کامنت: {str(e)}", 500)


@api_view(["POST"])
def get_comments(request):
    """
    API endpoint to retrieve comments
    Required fields: comment_api_key
    Only clients with valid API key can view comments.
    """
    try:
        # Validate required fields
        RequsetChecker(request.POST, [
            {
                "name": "comment_api_key",
                "format": "^[a-zA-Z0-9_]{20,64}$"
            }
        ], request)

        comment_api_key = str(request.POST["comment_api_key"])
        
        # Validate comment API key
        if not validate_comment_api_key(comment_api_key):
            return myResponse.Error("Invalid comment API key", Errors.InvalidArgument.code)

        # Get all active comments
        comments = Comment.objects.filter(is_active=True).select_related('user')

        # Format response data
        comments_data = []
        for comment in comments:
            comment_data = {
                "id": comment.id,
                "content": comment.content,
                "created_at": str(comment.created_at),
                "updated_at": str(comment.updated_at),
                "user": {
                    "identifier": comment.user.identifier,
                    "full_name": comment.user.profile.first_name if hasattr(comment.user, 'profile') else "",
                }
            }
            comments_data.append(comment_data)

        return myResponse.OK("کامنت‌ها با موفقیت دریافت شد", comments_data)

    except Exception as e:
        return myResponse.Error(f"خطا در دریافت کامنت‌ها: {str(e)}", 500)


@api_view(["POST"])
def get_user_comments(request):
    """
    API endpoint to get comments for a specific user
    Required fields: token, phone_number
    """
    try:
        # Validate required fields
        RequsetChecker(request.POST, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            },
        ], request)

        token = str(request.POST["token"])

        # Validate user
        u = User.objects.filter(token=token)
        if not u.exists():
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)
        user = u[0]

        # Get user's comments
        comments = Comment.objects.filter(user=user, is_active=True)

        # Format response data
        comments_data = []
        for comment in comments:
            comment_data = {
                "id": comment.id,
                "content": comment.content,
                "created_at": str(comment.created_at),
                "updated_at": str(comment.updated_at)
            }
            comments_data.append(comment_data)

        return myResponse.OK("کامنت‌های کاربر با موفقیت دریافت شد", comments_data)

    except Exception as e:
        return myResponse.Error(f"خطا در دریافت کامنت‌های کاربر: {str(e)}", 500) 