from rest_framework.decorators import api_view
from rest_framework.response import Response
from django.contrib.auth import get_user_model
from django.db.models import Q, Count, Avg
from django.utils import timezone
from datetime import datetime, timedelta
from .models import Category, Resource, VideoResource, PDFResource, ImageSeriesResource, ImageSeriesItem, UserResourceInteraction, UserCategoryProgress, User
from .views.views import myResponse, RequsetChecker, Errors, writeErrorToLog
from django.contrib.auth.models import Group

# Helper functions
def get_user_from_token(token):
    """Get user from token, following existing pattern"""
    try:
        user = User.objects.get(token=token)
        return user
    except User.DoesNotExist:
        return None

def is_doctor(user):
    """Check if user is a doctor"""
    if not user or not user.django_user:
        return False
    return user.django_user.groups.filter(name='doctor').exists()

# Category APIs
@api_view(["GET", "POST"])
def categories(request):
    if request.method == "POST":
        try:
            RequsetChecker(request.POST, [
                {
                    "name": "token",
                    "format": "^(\S){30}$"
                },
                {
                    "name": "name",
                    "format": "^.+$",
                    "errorMessage": "نام دسته‌بندی الزامی است"
                },
                {
                    "name": "description",
                    "format": "^.*$",
                    "required": False
                },
                {
                    "name": "parent_id",
                    "format": "^[\d]*$",
                    "required": False
                },
                {
                    "name": "ordering",
                    "format": "^[\d]*$",
                    "required": False
                }
            ], request)

            token = request.POST["token"]
            user = get_user_from_token(token)
            if not user:
                return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

            if not is_doctor(user):
                return myResponse.Error("فقط پزشکان می‌توانند دسته‌بندی ایجاد کنند", 403)

            name = request.POST["name"]
            description = request.POST.get("description", "")
            parent_id = request.POST.get("parent_id")
            ordering = int(request.POST.get("ordering", 0))

            # Create category
            category_data = {
                "name": name,
                "description": description,
                "ordering": ordering,
                "creator": user.django_user
            }

            if parent_id:
                try:
                    parent = Category.objects.get(id=parent_id)
                    category_data["parent"] = parent
                except Category.DoesNotExist:
                    return myResponse.Error("دسته‌بندی والد یافت نشد", 404)

            category = Category.objects.create(**category_data)

            data = {
                "id": category.id,
                "name": category.name,
                "slug": category.slug,
                "description": category.description,
                "parent_id": category.parent.id if category.parent else None,
                "ordering": category.ordering,
                "is_active": category.is_active,
                "created_at": int(category.created_at.timestamp())
            }

            return myResponse.OK("دسته‌بندی با موفقیت ایجاد شد", data)

        except ValueError as e:
            writeErrorToLog("categories ---> POST\n" + str(e.args), request)
            return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
        except Exception as e:
            writeErrorToLog("categories ---> POST\n" + str(e.args), request)
            return myResponse.Error(str(e), Errors.InternalError.code)

    else:  # GET
        try:
            RequsetChecker(request.GET, [
                {
                    "name": "token",
                    "format": "^(\S){30}$"
                }
            ], request)

            token = request.GET["token"]
            user = get_user_from_token(token)
            if not user:
                return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

            # Get root categories (no parent) with subcategories
            categories = Category.objects.filter(parent=None, is_active=True).order_by('ordering', 'name')
            
            data = []
            for category in categories:
                category_data = {
                    "id": category.id,
                    "name": category.name,
                    "slug": category.slug,
                    "description": category.description,
                    "resource_count": category.resource_count,
                    "subcategories": []
                }

                # Get subcategories
                subcategories = category.subcategories.filter(is_active=True).order_by('ordering', 'name')
                for subcategory in subcategories:
                    subcategory_data = {
                        "id": subcategory.id,
                        "name": subcategory.name,
                        "slug": subcategory.slug,
                        "description": subcategory.description,
                        "resource_count": subcategory.resource_count
                    }
                    category_data["subcategories"].append(subcategory_data)

                data.append(category_data)

            return myResponse.OK("لیست دسته‌بندی‌ها", data)

        except ValueError as e:
            writeErrorToLog("categories ---> GET\n" + str(e.args), request)
            return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
        except Exception as e:
            writeErrorToLog("categories ---> GET\n" + str(e.args), request)
            return myResponse.Error(str(e), Errors.InternalError.code)

@api_view(["GET", "PUT", "DELETE"])
def category_detail(request, category_id):
    try:
        RequsetChecker(request.GET if request.method == "GET" else request.POST, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            }
        ], request)

        token = request.GET.get("token") or request.POST.get("token")
        user = get_user_from_token(token)
        if not user:
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        try:
            category = Category.objects.get(id=category_id)
        except Category.DoesNotExist:
            return myResponse.Error("دسته‌بندی یافت نشد", 404)

        if request.method == "GET":
            # Check if user can access this category
            if not category.is_active and not is_doctor(user):
                return myResponse.Error("دسته‌بندی غیرفعال است", 403)

            data = {
                "id": category.id,
                "name": category.name,
                "slug": category.slug,
                "description": category.description,
                "parent_id": category.parent.id if category.parent else None,
                "ordering": category.ordering,
                "is_active": category.is_active,
                "resource_count": category.resource_count,
                "created_at": int(category.created_at.timestamp()),
                "updated_at": int(category.updated_at.timestamp())
            }

            return myResponse.OK("جزئیات دسته‌بندی", data)

        elif request.method == "PUT":
            # Only doctors can edit categories
            if not is_doctor(user):
                return myResponse.Error("فقط پزشکان می‌توانند دسته‌بندی‌ها را ویرایش کنند", 403)

            # Only creator can edit
            if category.creator != user.django_user:
                return myResponse.Error("شما مجاز به ویرایش این دسته‌بندی نیستید", 403)

            # Update fields
            if "name" in request.POST:
                category.name = request.POST["name"]
            if "description" in request.POST:
                category.description = request.POST["description"]
            if "ordering" in request.POST:
                category.ordering = int(request.POST["ordering"])
            if "is_active" in request.POST:
                category.is_active = bool(int(request.POST["is_active"]))

            category.save()

            data = {
                "id": category.id,
                "name": category.name,
                "slug": category.slug,
                "description": category.description,
                "ordering": category.ordering,
                "is_active": category.is_active
            }

            return myResponse.OK("دسته‌بندی با موفقیت بروزرسانی شد", data)

        elif request.method == "DELETE":
            # Only doctors can delete categories
            if not is_doctor(user):
                return myResponse.Error("فقط پزشکان می‌توانند دسته‌بندی‌ها را حذف کنند", 403)

            # Only creator can delete
            if category.creator != user.django_user:
                return myResponse.Error("شما مجاز به حذف این دسته‌بندی نیستید", 403)

            # Check if category has resources
            if category.resources.exists():
                return myResponse.Error("نمی‌توان دسته‌بندی دارای محتوا را حذف کرد", 400)

            category.delete()
            return myResponse.OK("دسته‌بندی با موفقیت حذف شد", {})

    except ValueError as e:
        writeErrorToLog("category_detail ---> " + request.method + "\n" + str(e.args), request)
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
    except Exception as e:
        writeErrorToLog("category_detail ---> " + request.method + "\n" + str(e.args), request)
        return myResponse.Error(str(e), Errors.InternalError.code)

# Resource APIs
@api_view(["GET", "POST"])
def resources(request):
    if request.method == "POST":
        try:
            RequsetChecker(request.POST, [
                {
                    "name": "token",
                    "format": "^(\S){30}$"
                },
                {
                    "name": "title",
                    "format": "^.+$",
                    "errorMessage": "عنوان محتوا الزامی است"
                },
                {
                    "name": "description",
                    "format": "^.*$",
                    "required": False
                },
                {
                    "name": "category_id",
                    "format": "^[\d]+$"
                },
                {
                    "name": "type",
                    "format": "^(video|pdf|image_series)$",
                    "errorMessage": "نوع محتوا باید video، pdf یا image_series باشد"
                },
                {
                    "name": "ordering",
                    "format": "^[\d]*$",
                    "required": False
                }
            ], request)

            token = request.POST["token"]
            user = get_user_from_token(token)
            if not user:
                return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

            if not is_doctor(user):
                return myResponse.Error("فقط پزشکان می‌توانند محتوا ایجاد کنند", 403)

            title = request.POST["title"]
            description = request.POST.get("description", "")
            category_id = int(request.POST["category_id"])
            resource_type = request.POST["type"]
            ordering = int(request.POST.get("ordering", 0))

            # Check if category exists and user owns it
            try:
                category = Category.objects.get(id=category_id)
                if category.creator != user.django_user:
                    return myResponse.Error("شما مجاز به ایجاد محتوا در این دسته‌بندی نیستید", 403)
            except Category.DoesNotExist:
                return myResponse.Error("دسته‌بندی یافت نشد", 404)

            # Create resource
            resource = Resource.objects.create(
                title=title,
                description=description,
                category=category,
                type=resource_type,
                ordering=ordering,
                creator=user.django_user
            )

            data = {
                "id": resource.id,
                "title": resource.title,
                "slug": resource.slug,
                "description": resource.description,
                "category_id": resource.category.id,
                "type": resource.type,
                "is_published": resource.is_published,
                "is_featured": resource.is_featured,
                "view_count": resource.view_count,
                "ordering": resource.ordering,
                "created_at": int(resource.created_at.timestamp())
            }

            return myResponse.OK("محتوا با موفقیت ایجاد شد", data)

        except ValueError as e:
            writeErrorToLog("resources ---> POST\n" + str(e.args), request)
            return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
        except Exception as e:
            writeErrorToLog("resources ---> POST\n" + str(e.args), request)
            return myResponse.Error(str(e), Errors.InternalError.code)

    else:  # GET
        try:
            RequsetChecker(request.GET, [
                {
                    "name": "token",
                    "format": "^(\S){30}$"
                }
            ], request)

            token = request.GET["token"]
            user = get_user_from_token(token)
            if not user:
                return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

            # Get query parameters
            # category_slug: Filter resources by category slug (instead of category_id)
            category_slug = request.GET.get("category_slug")
            resource_type = request.GET.get("type")
            is_featured = request.GET.get("is_featured")
            search = request.GET.get("search")

            # Build queryset
            queryset = Resource.objects.filter(is_published=True)

            if category_slug:
                queryset = queryset.filter(category__slug=category_slug)
            if resource_type:
                queryset = queryset.filter(type=resource_type)
            if is_featured:
                queryset = queryset.filter(is_featured=True)
            if search:
                queryset = queryset.filter(
                    Q(title__icontains=search) | Q(description__icontains=search)
                )

            # Order by featured, then ordering, then title
            queryset = queryset.order_by('-is_featured', 'ordering', 'title')

            data = []
            for resource in queryset:
                # Get user interaction status for this resource
                try:
                    interaction = UserResourceInteraction.objects.get(
                        user=user.django_user,
                        resource=resource
                    )
                    user_status = {
                        "viewed": interaction.viewed,
                        "completed": interaction.completed,
                        "bookmarked": interaction.bookmarked,
                        "liked": interaction.liked,
                        "progress": interaction.progress,
                        "first_viewed_at": int(interaction.first_viewed_at.timestamp()) if interaction.first_viewed_at else None,
                        "last_viewed_at": int(interaction.last_viewed_at.timestamp()) if interaction.last_viewed_at else None,
                        "completed_at": int(interaction.completed_at.timestamp()) if interaction.completed_at else None
                    }
                except UserResourceInteraction.DoesNotExist:
                    # User has no interaction with this resource
                    user_status = {
                        "viewed": False,
                        "completed": False,
                        "bookmarked": False,
                        "liked": False,
                        "progress": 0,
                        "first_viewed_at": None,
                        "last_viewed_at": None,
                        "completed_at": None
                    }

                resource_data = {
                    "id": resource.id,
                    "title": resource.title,
                    "slug": resource.slug,
                    "description": resource.description,
                    "category_id": resource.category.id,
                    "category_name": resource.category.name,
                    "type": resource.type,
                    "is_featured": resource.is_featured,
                    "view_count": resource.view_count,
                    "created_at": int(resource.created_at.timestamp()),
                    "user_status": user_status
                }

                # Add type-specific data
                if resource.type == "video" and hasattr(resource, 'video'):
                    resource_data["video"] = {
                        "video_url": resource.video.video_url,
                        "duration_minutes": resource.video.duration_minutes,
                        "thumbnail": resource.video.thumbnail.url if resource.video.thumbnail else None
                    }
                elif resource.type == "pdf" and hasattr(resource, 'pdf'):
                    resource_data["pdf"] = {
                        "file": resource.pdf.file.url,
                        "file_size": resource.pdf.file_size,
                        "page_count": resource.pdf.page_count
                    }
                elif resource.type == "image_series" and hasattr(resource, 'image_series'):
                    items = resource.image_series.items.all().order_by('ordering')
                    resource_data["image_series"] = {
                        "items_count": items.count(),
                        "items": [{
                            "id": item.id,
                            "image": item.image.url,
                            "caption": item.caption,
                            "ordering": item.ordering
                        } for item in items]
                    }

                data.append(resource_data)

            return myResponse.OK("لیست محتوا", data)

        except ValueError as e:
            writeErrorToLog("resources ---> GET\n" + str(e.args), request)
            return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
        except Exception as e:
            writeErrorToLog("resources ---> GET\n" + str(e.args), request)
            return myResponse.Error(str(e), Errors.InternalError.code)

@api_view(["GET"])
def resources_by_category_name(request, category_name):
    try:
        RequsetChecker(request.GET, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            }
        ], request)

        token = request.GET["token"]
        user = get_user_from_token(token)
        if not user:
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        # Find category by name (case-insensitive) or slug fallback
        try:
            category = Category.objects.get(Q(name__iexact=category_name) | Q(slug__iexact=category_name))
        except Category.DoesNotExist:
            return myResponse.Error("دسته‌بندی یافت نشد", 404)

        # Optional filters
        resource_type = request.GET.get("type")
        is_featured = request.GET.get("is_featured")
        search = request.GET.get("search")

        queryset = Resource.objects.filter(is_published=True, category=category)
        if resource_type:
            queryset = queryset.filter(type=resource_type)
        if is_featured:
            queryset = queryset.filter(is_featured=True)
        if search:
            queryset = queryset.filter(
                Q(title__icontains=search) | Q(description__icontains=search)
            )

        queryset = queryset.order_by('-is_featured', 'ordering', 'title')

        data = []
        for resource in queryset:
            # Get user interaction status for this resource
            try:
                interaction = UserResourceInteraction.objects.get(
                    user=user.django_user,
                    resource=resource
                )
                user_status = {
                    "viewed": interaction.viewed,
                    "completed": interaction.completed,
                    "bookmarked": interaction.bookmarked,
                    "liked": interaction.liked,
                    "progress": interaction.progress,
                    "first_viewed_at": int(interaction.first_viewed_at.timestamp()) if interaction.first_viewed_at else None,
                    "last_viewed_at": int(interaction.last_viewed_at.timestamp()) if interaction.last_viewed_at else None,
                    "completed_at": int(interaction.completed_at.timestamp()) if interaction.completed_at else None
                }
            except UserResourceInteraction.DoesNotExist:
                # User has no interaction with this resource
                user_status = {
                    "viewed": False,
                    "completed": False,
                    "bookmarked": False,
                    "liked": False,
                    "progress": 0,
                    "first_viewed_at": None,
                    "last_viewed_at": None,
                    "completed_at": None
                }

            resource_data = {
                "id": resource.id,
                "title": resource.title,
                "slug": resource.slug,
                "description": resource.description,
                "category_id": resource.category.id,
                "category_name": resource.category.name,
                "type": resource.type,
                "is_featured": resource.is_featured,
                "view_count": resource.view_count,
                "created_at": int(resource.created_at.timestamp()),
                "user_status": user_status
            }

            # Add type-specific data
            if resource.type == "video" and hasattr(resource, 'video'):
                resource_data["video"] = {
                    "video_url": resource.video.video_url,
                    "duration_minutes": resource.video.duration_minutes,
                    "thumbnail": resource.video.thumbnail.url if resource.video.thumbnail else None
                }
            elif resource.type == "pdf" and hasattr(resource, 'pdf'):
                resource_data["pdf"] = {
                    "file": resource.pdf.file.url,
                    "file_size": resource.pdf.file_size,
                    "page_count": resource.pdf.page_count
                }
            elif resource.type == "image_series" and hasattr(resource, 'image_series'):
                items = resource.image_series.items.all().order_by('ordering')
                resource_data["image_series"] = {
                    "items_count": items.count(),
                    "items": [{
                        "id": item.id,
                        "image": item.image.url,
                        "caption": item.caption,
                        "ordering": item.ordering
                    } for item in items]
                }

            data.append(resource_data)

        return myResponse.OK("لیست محتوا", data)

    except ValueError as e:
        writeErrorToLog("resources_by_category_name ---> GET\n" + str(e.args), request)
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
    except Exception as e:
        writeErrorToLog("resources_by_category_name ---> GET\n" + str(e.args), request)
        return myResponse.Error(str(e), Errors.InternalError.code)

@api_view(["GET", "PUT", "DELETE"])
def resource_detail(request, resource_id):
    try:
        RequsetChecker(request.GET if request.method == "GET" else request.POST, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            }
        ], request)

        token = request.GET.get("token") or request.POST.get("token")
        user = get_user_from_token(token)
        if not user:
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        try:
            resource = Resource.objects.get(id=resource_id)
        except Resource.DoesNotExist:
            return myResponse.Error("محتوا یافت نشد", 404)

        if request.method == "GET":
            # Check if user can access this resource
            if not resource.is_published and not is_doctor(user):
                return myResponse.Error("این محتوا منتشر نشده است", 403)

            # Increment view count
            resource.view_count += 1
            resource.save()

            # Mark as viewed for user
            interaction, created = UserResourceInteraction.objects.get_or_create(
                user=user.django_user,
                resource=resource,
                defaults={
                    "viewed": True,
                    "first_viewed_at": timezone.now(),
                    "last_viewed_at": timezone.now()
                }
            )
            
            if not created:
                # Update existing interaction
                interaction.viewed = True
                interaction.last_viewed_at = timezone.now()
                interaction.save()

            # Get user interaction status
            user_status = {
                "viewed": interaction.viewed,
                "completed": interaction.completed,
                "bookmarked": interaction.bookmarked,
                "liked": interaction.liked,
                "progress": interaction.progress,
                "first_viewed_at": interaction.first_viewed_at.isoformat() if interaction.first_viewed_at else None,
                "last_viewed_at": interaction.last_viewed_at.isoformat() if interaction.last_viewed_at else None,
                "completed_at": interaction.completed_at.isoformat() if interaction.completed_at else None
            }

            data = {
                "id": resource.id,
                "title": resource.title,
                "slug": resource.slug,
                "description": resource.description,
                "category_id": resource.category.id,
                "category_name": resource.category.name,
                "type": resource.type,
                "is_featured": resource.is_featured,
                "view_count": resource.view_count,
                "created_at": int(resource.created_at.timestamp()),
                "updated_at": int(resource.updated_at.timestamp()),
                "user_status": user_status
            }

            # Add type-specific data
            if resource.type == "video" and hasattr(resource, 'video'):
                data["video"] = {
                    "video_url": resource.video.video_url,
                    "duration_minutes": resource.video.duration_minutes,
                    "thumbnail": resource.video.thumbnail.url if resource.video.thumbnail else None
                }
            elif resource.type == "pdf" and hasattr(resource, 'pdf'):
                data["pdf"] = {
                    "file": resource.pdf.file.url,
                    "file_size": resource.pdf.file_size,
                    "page_count": resource.pdf.page_count
                }
            elif resource.type == "image_series" and hasattr(resource, 'image_series'):
                items = resource.image_series.items.all().order_by('ordering')
                data["image_series"] = {
                    "items_count": items.count(),
                    "items": [{
                        "id": item.id,
                        "image": item.image.url,
                        "caption": item.caption,
                        "ordering": item.ordering
                    } for item in items]
                }

            return myResponse.OK("جزئیات محتوا", data)

        elif request.method == "PUT":
            # Only doctors can edit resources
            if not is_doctor(user):
                return myResponse.Error("فقط پزشکان می‌توانند محتوا را ویرایش کنند", 403)

            # Only creator can edit
            if resource.creator != user.django_user:
                return myResponse.Error("شما مجاز به ویرایش این محتوا نیستید", 403)

            # Update fields
            if "title" in request.POST:
                resource.title = request.POST["title"]
            if "description" in request.POST:
                resource.description = request.POST["description"]
            if "ordering" in request.POST:
                resource.ordering = int(request.POST["ordering"])
            if "is_published" in request.POST:
                resource.is_published = bool(int(request.POST["is_published"]))
            if "is_featured" in request.POST:
                resource.is_featured = bool(int(request.POST["is_featured"]))

            resource.save()

            data = {
                "id": resource.id,
                "title": resource.title,
                "slug": resource.slug,
                "description": resource.description,
                "is_published": resource.is_published,
                "is_featured": resource.is_featured
            }

            return myResponse.OK("محتوا با موفقیت بروزرسانی شد", data)

        elif request.method == "DELETE":
            # Only doctors can delete resources
            if not is_doctor(user):
                return myResponse.Error("فقط پزشکان می‌توانند محتوا را حذف کنند", 403)

            # Only creator can delete
            if resource.creator != user.django_user:
                return myResponse.Error("شما مجاز به حذف این محتوا نیستید", 403)

            resource.delete()
            return myResponse.OK("محتوا با موفقیت حذف شد", {})

    except ValueError as e:
        writeErrorToLog("resource_detail ---> " + request.method + "\n" + str(e.args), request)
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
    except Exception as e:
        writeErrorToLog("resource_detail ---> " + request.method + "\n" + str(e.args), request)
        return myResponse.Error(str(e), Errors.InternalError.code)

# User Interaction APIs
@api_view(["POST"])
def mark_resource_interaction(request):
    try:
        RequsetChecker(request.POST, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            },
            {
                "name": "resource_id",
                "format": "^[\d]+$"
            },
            {
                "name": "interaction_type",
                "format": "^(viewed|completed|bookmarked|liked)$",
                "errorMessage": "نوع تعامل باید viewed، completed، bookmarked یا liked باشد"
            },
            {
                "name": "progress",
                "format": "^[0-9]+\.?[0-9]*$",
                "required": False
            },
            {
                "name": "time_spent_seconds",
                "format": "^[\d]*$",
                "required": False
            }
        ], request)

        token = request.POST["token"]
        user = get_user_from_token(token)
        if not user:
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        resource_id = int(request.POST["resource_id"])
        interaction_type = request.POST["interaction_type"]
        progress = float(request.POST.get("progress", 0))
        time_spent_seconds = int(request.POST.get("time_spent_seconds", 0))

        try:
            resource = Resource.objects.get(id=resource_id, is_published=True)
        except Resource.DoesNotExist:
            return myResponse.Error("محتوا یافت نشد", 404)

        # Get or create interaction
        interaction, created = UserResourceInteraction.objects.get_or_create(
            user=user.django_user,
            resource=resource,
            defaults={
                "progress": progress,
                "time_spent_seconds": time_spent_seconds,
                "first_viewed_at": timezone.now(),
                "last_viewed_at": timezone.now()
            }
        )

        if not created:
            # Update existing interaction
            interaction.progress = max(interaction.progress, progress)
            interaction.time_spent_seconds += time_spent_seconds
            interaction.last_viewed_at = timezone.now()

        # Update interaction based on type
        if interaction_type == "viewed":
            interaction.viewed = True
        elif interaction_type == "completed":
            interaction.completed = True
            interaction.completed_at = timezone.now()
        elif interaction_type == "bookmarked":
            interaction.bookmarked = not interaction.bookmarked  # Toggle bookmark
        elif interaction_type == "liked":
            interaction.liked = not interaction.liked  # Toggle like

        interaction.save()

        # Update category progress
        update_category_progress(user.django_user, resource.category)

        data = {
            "interaction_id": interaction.id,
            "progress": interaction.progress,
            "time_spent_seconds": interaction.time_spent_seconds,
            "completed_at": int(interaction.completed_at.timestamp()) if interaction.completed_at else None
        }

        return myResponse.OK("تعامل با موفقیت ثبت شد", data)

    except ValueError as e:
        writeErrorToLog("mark_resource_interaction ---> POST\n" + str(e.args), request)
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
    except Exception as e:
        writeErrorToLog("mark_resource_interaction ---> POST\n" + str(e.args), request)
        return myResponse.Error(str(e), Errors.InternalError.code)

def update_category_progress(user, category):
    """Update user's progress for a category"""
    # Get all resources in category
    resources = category.resources.filter(is_published=True)
    total_resources = resources.count()
    
    if total_resources == 0:
        return

    # Get user interactions
    viewed_resources = UserResourceInteraction.objects.filter(
        user=user,
        resource__in=resources,
        viewed=True
    ).values_list('resource_id', flat=True).distinct()

    completed_resources = UserResourceInteraction.objects.filter(
        user=user,
        resource__in=resources,
        completed=True
    ).values_list('resource_id', flat=True).distinct()

    viewed_count = len(viewed_resources)
    completed_count = len(completed_resources)
    completion_percent = (completed_count / total_resources) * 100 if total_resources > 0 else 0

    # Update or create category progress
    progress, created = UserCategoryProgress.objects.get_or_create(
        user=user,
        category=category,
        defaults={
            "total_resources": total_resources,
            "viewed_resources": viewed_count,
            "completed_resources": completed_count,
            "completion_percent": completion_percent
        }
    )

    if not created:
        progress.total_resources = total_resources
        progress.viewed_resources = viewed_count
        progress.completed_resources = completed_count
        progress.completion_percent = completion_percent
        progress.save()

# Content Discovery APIs
@api_view(["GET"])
def unseen_resources(request):
    try:
        RequsetChecker(request.GET, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            }
        ], request)

        token = request.GET["token"]
        user = get_user_from_token(token)
        if not user:
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        # Get resources user hasn't viewed
        viewed_resources = UserResourceInteraction.objects.filter(
            user=user.django_user,
            viewed=True
        ).values_list('resource_id', flat=True)

        unseen_resources = Resource.objects.filter(
            is_published=True
        ).exclude(
            id__in=viewed_resources
        ).order_by('-is_featured', 'ordering', 'title')

        data = []
        for resource in unseen_resources:
            # Get user interaction status for this resource
            try:
                interaction = UserResourceInteraction.objects.get(
                    user=user.django_user,
                    resource=resource
                )
                user_status = {
                    "viewed": interaction.viewed,
                    "completed": interaction.completed,
                    "bookmarked": interaction.bookmarked,
                    "liked": interaction.liked,
                    "progress": interaction.progress,
                    "first_viewed_at": int(interaction.first_viewed_at.timestamp()) if interaction.first_viewed_at else None,
                    "last_viewed_at": int(interaction.last_viewed_at.timestamp()) if interaction.last_viewed_at else None,
                    "completed_at": int(interaction.completed_at.timestamp()) if interaction.completed_at else None
                }
            except UserResourceInteraction.DoesNotExist:
                # User has no interaction with this resource
                user_status = {
                    "viewed": False,
                    "completed": False,
                    "bookmarked": False,
                    "liked": False,
                    "progress": 0,
                    "first_viewed_at": None,
                    "last_viewed_at": None,
                    "completed_at": None
                }

            resource_data = {
                "id": resource.id,
                "title": resource.title,
                "slug": resource.slug,
                "description": resource.description,
                "category_id": resource.category.id,
                "category_name": resource.category.name,
                "type": resource.type,
                "is_featured": resource.is_featured,
                "view_count": resource.view_count,
                "created_at": int(resource.created_at.timestamp()),
                "user_status": user_status
            }
            data.append(resource_data)

        return myResponse.OK("محتواهای دیده نشده", data)

    except ValueError as e:
        writeErrorToLog("unseen_resources ---> GET\n" + str(e.args), request)
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
    except Exception as e:
        writeErrorToLog("unseen_resources ---> GET\n" + str(e.args), request)
        return myResponse.Error(str(e), Errors.InternalError.code)

@api_view(["GET"])
def featured_resources(request):
    try:
        RequsetChecker(request.GET, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            }
        ], request)

        token = request.GET["token"]
        user = get_user_from_token(token)
        if not user:
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        featured_resources = Resource.objects.filter(
            is_published=True,
            is_featured=True
        ).order_by('ordering', 'title')

        data = []
        for resource in featured_resources:
            # Get user interaction status for this resource
            try:
                interaction = UserResourceInteraction.objects.get(
                    user=user.django_user,
                    resource=resource
                )
                user_status = {
                    "viewed": interaction.viewed,
                    "completed": interaction.completed,
                    "bookmarked": interaction.bookmarked,
                    "liked": interaction.liked,
                    "progress": interaction.progress,
                    "first_viewed_at": int(interaction.first_viewed_at.timestamp()) if interaction.first_viewed_at else None,
                    "last_viewed_at": int(interaction.last_viewed_at.timestamp()) if interaction.last_viewed_at else None,
                    "completed_at": int(interaction.completed_at.timestamp()) if interaction.completed_at else None
                }
            except UserResourceInteraction.DoesNotExist:
                # User has no interaction with this resource
                user_status = {
                    "viewed": False,
                    "completed": False,
                    "bookmarked": False,
                    "liked": False,
                    "progress": 0,
                    "first_viewed_at": None,
                    "last_viewed_at": None,
                    "completed_at": None
                }

            resource_data = {
                "id": resource.id,
                "title": resource.title,
                "slug": resource.slug,
                "description": resource.description,
                "category_id": resource.category.id,
                "category_name": resource.category.name,
                "type": resource.type,
                "view_count": resource.view_count,
                "created_at": int(resource.created_at.timestamp()),
                "user_status": user_status
            }
            data.append(resource_data)

        return myResponse.OK("محتواهای ویژه", data)

    except ValueError as e:
        writeErrorToLog("featured_resources ---> GET\n" + str(e.args), request)
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
    except Exception as e:
        writeErrorToLog("featured_resources ---> GET\n" + str(e.args), request)
        return myResponse.Error(str(e), Errors.InternalError.code)

@api_view(["GET"])
def user_progress(request):
    try:
        RequsetChecker(request.GET, [
            {
                "name": "token",
                "format": "^(\S){30}$"
            }
        ], request)

        token = request.GET["token"]
        user = get_user_from_token(token)
        if not user:
            return myResponse.Error(Errors.InvalidToken.message, Errors.InvalidToken.code)

        # Get user's category progress
        category_progress = UserCategoryProgress.objects.filter(user=user.django_user)

        data = []
        for progress in category_progress:
            progress_data = {
                "category_id": progress.category.id,
                "category_name": progress.category.name,
                "total_resources": progress.total_resources,
                "viewed_resources": progress.viewed_resources,
                "completed_resources": progress.completed_resources,
                "completion_percent": progress.completion_percent,
                "updated_at": int(progress.updated_at.timestamp())
            }
            data.append(progress_data)

        return myResponse.OK("پیشرفت کاربر", data)

    except ValueError as e:
        writeErrorToLog("user_progress ---> GET\n" + str(e.args), request)
        return myResponse.Error(e.args[0], Errors.InvalidArgument.code)
    except Exception as e:
        writeErrorToLog("user_progress ---> GET\n" + str(e.args), request)
        return myResponse.Error(str(e), Errors.InternalError.code) 