from collections import OrderedDict
from neo4j import GraphDatabase
import traceback
import APIs.modules.scoring_module as scoring_module   # Custom module for scoring
import random
import math 
import json

NEO4J_URI = "bolt://localhost:7687"
NEO4J_USER = "neo4j"
NEO4J_PASSWORD = "mahila1234"
MEAL_TYPE_TO_MO = {"LUNCH": "MO-LU", "DINNER": "MO-DI", "BREAKFAST": "MO-BR", "SNACK": "MO-SN"}
COMPLEX_PRIORITY_MEALS = ["LUNCH", "DINNER"]
POSSIBLE_MEAL_RELATIONSHIPS = ["LUNCH", "DINNER", "BREAKFAST"] # PAIRED_WITH is also used but not for primary meal type relationship
MEAL_TYPE_TO_SCORE_CATEGORY = {"BREAKFAST": "main", "LUNCH": "main", "DINNER": "main", "SNACK": "snacks"}

calorie_distributions = {
    1200: {"B": 250, "L": 350, "D": 300, "S1": 150, "S2": 150},
    1300: {"B": 300, "L": 350, "D": 350, "S1": 150, "S2": 150},
    1400: {"B": 350, "L": 400, "D": 350, "S1": 150, "S2": 150},
    1500: {"B": 400, "L": 400, "D": 400, "S1": 150, "S2": 150},
    1600: {"B": 400, "L": 450, "D": 400, "S1": 150, "S2": 200},
    1700: {"B": 400, "L": 500, "D": 450, "S1": 150, "S2": 200},
    1800: {"B": 400, "L": 550, "D": 450, "S1": 200, "S2": 200},
    1900: {"B": 450, "L": 550, "D": 500, "S1": 200, "S2": 200},
    2000: {"B": 500, "L": 600, "D": 500, "S1": 200, "S2": 200},
    2100: {"B": 500, "L": 600, "D": 550, "S1": 200, "S2": 250},
    2200: {"B": 500, "L": 650, "D": 600, "S1": 200, "S2": 250},
    2300: {"B": 550, "L": 650, "D": 600, "S1": 250, "S2": 250},
    2400: {"B": 550, "L": 700, "D": 600, "S1": 250, "S2": 300}
}
MEAL_TYPE_TO_DIST_KEY = {"BREAKFAST": "B", "LUNCH": "L", "DINNER": "D", "SNACK": "S1"} # S2 for snacks is not directly used here for target calories

nutrient_keys_map = {
    'calories': ("Calories", "Energy"),
    'protein': ("Protein", "Protein_g"),
    'carbohydrate': ("Carbohydrate", "Carbohydrates_g"),
    'dietary_fiber': ("Dietary Fiber", "Fiber_g"),
    'total_sugars': ("Total Sugars", "Sugar_g"),
    'added_sugar_g': ("AddedSugar_g",),
    'total_saturated': ("Total Saturated", "SaturatedFat_g"),
    'sodium': ("Sodium", "Sodium_mg"),
    'potassium': ("Potassium", "Potassium_mg"),
    'total_monounsaturated': ("Total Monounsaturated", "MonounsaturatedFat_g"),
    'total_polyunsaturated': ("Total Polyunsaturated", "PolyunsaturatedFat_g"),
    'total_fat': ("Total Fat", "Fat_g"),
    'omega_3': ("Omega-3",),
    'cholesterol': ("Cholesterol", "Cholesterol_mg"),
    'total_trans': ("Total Trans", "TransFat_g"),
    'gl': ("GL", "glycemic_load"),
    'energy': ("Energy", "Calories") # Duplicate for flexibility
}

def round_cr(cr):
    """Rounds calorie requirement to the nearest 100."""
    if cr is None:
        return None
    try:
        cr_float = float(cr)
        base = (int(cr_float) // 100) * 100
        remainder = cr_float - base
        return int(base + 100) if remainder > 50.0 else int(base)
    except (ValueError, TypeError):
        print(f"Warning: Could not round CR: {cr}")
        return None

# --- Helper Functions ---
def _get_assigned_mcs(food_dict):
    """Extracts AssignedMC values as a set from a food dictionary."""
    mc_string = food_dict.get('AssignedMC')
    return {val.strip() for val in mc_string.split(',') if val.strip()} if mc_string else set()

def _build_filter_clauses(target_alias, params, req_cond, req_aller, excl_cond, excl_aller, other_filters, input_mc_list_for_exclusion=None, meal_type_for_mc_exclusion=None):
    """Builds WHERE clauses for Neo4j queries based on filters."""
    where_clauses = []
    filter_count = len(params) # Start counting params from existing ones
    filters = other_filters if other_filters else {}

    for key, value in filters.items():
        param_name = f"filterParam{filter_count}"
        prop_ref = f"{target_alias}.{key}"
        if isinstance(value, list): # For properties that can have multiple comma-separated values
            where_clauses.append(f"ANY(item IN ${param_name} WHERE item IN [val IN split(coalesce({prop_ref}, ''), ',') | trim(val)])")
            params[param_name] = [str(v) for v in value] # Ensure values are strings
        else:
            where_clauses.append(f"{prop_ref} = ${param_name}")
            params[param_name] = str(value) # Ensure value is a string
        filter_count += 1

    if req_cond:
        req_cond_param = f"reqCondList{filter_count}"
        params[req_cond_param] = req_cond
        where_clauses.append(f"EXISTS {{ MATCH ({target_alias})-[]-(c:Condition) WHERE c.name IN ${req_cond_param} }}")
        filter_count += 1

    if req_aller:
        req_aller_param = f"reqAllerList{filter_count}"
        params[req_aller_param] = req_aller
        where_clauses.append(f"EXISTS {{ MATCH ({target_alias})-[]-(a:Allergen) WHERE a.name IN ${req_aller_param} }}")
        filter_count += 1

    if excl_cond:
        excl_cond_param = f"exclCondList{filter_count}"
        params[excl_cond_param] = excl_cond
        where_clauses.append(f"NOT EXISTS {{ MATCH ({target_alias})-[]-(c:Condition) WHERE c.name IN ${excl_cond_param} }}")
        filter_count += 1

    if excl_aller:
        excl_aller_param = f"exclAllerList{filter_count}"
        params[excl_aller_param] = excl_aller
        where_clauses.append(f"NOT EXISTS {{ MATCH ({target_alias})-[]-(a:Allergen) WHERE a.name IN ${excl_aller_param} }}")
        filter_count += 1

    # MC exclusion logic: if meal type is LUNCH or DINNER and there are MCs from input foods to exclude
    if meal_type_for_mc_exclusion in ["LUNCH", "DINNER"] and input_mc_list_for_exclusion:
        mc_excl_param = f"inputMcListParam{filter_count}"
        params[mc_excl_param] = input_mc_list_for_exclusion
        # Exclude foods that share any MC with the input MC list
        where_clauses.append(f"NONE(input_mc IN ${mc_excl_param} WHERE input_mc IN [val IN split(coalesce({target_alias}.AssignedMC, ''), ',') | trim(val)])")
        filter_count += 1
    return where_clauses

def _execute_recommendation_query(driver, query, params):
    """Executes a Neo4j query and returns results as a list of dictionaries."""
    results = []
    try:
        with driver.session(database="neo4j") as session:
            records = session.execute_read(lambda tx: list(tx.run(query, **params))) # Ensure all results are fetched
        for record in records:
            node = record.get("food") # Assuming the food node is aliased as 'food'
            if node:
                results.append(dict(node.items())) # Convert Node object to dictionary
            else:
                # This can happen if the query returns other things or the alias is different
                print(f"Warning: Query record missing 'food' alias: {record}")
    except Exception as e:
        print(f"--- Error during query execution ---")
        print(f"Error Type: {type(e).__name__}, Message: {e}")
        print(f"Failed Query: {query}")
        print(f"Failed Params: {json.dumps(params, indent=2)}")
        # traceback.print_exc() # Optionally print full traceback
    return results

def priority_return(meal_type_results_raw, input_food_dicts):
    """
    Filters meal type results based on priority MCs.
    It returns prioritized results if found and not in basic exclusion list.
    Otherwise, returns all raw results that are NOT in the basic exclusion list.
    NOTE: The main MC exclusion logic is now handled by _build_filter_clauses.
    """
    priority_mc_list = ["MC-SA", "MC-AP", "MC-BV"]  # MCs that get priority if available
    
    # We no longer derive input_mc_list for *exclusion* here for general filtering.
    # The actual exclusion logic for MCs is now done in _build_filter_clauses.
    # This function primarily focuses on *prioritizing* results.

    # Collect all MCs from the current sequence (input_food_dicts) to check against priority_mc_list
    # to avoid recommending a priority MC if it's *already* in the sequence.
    # This is a different check than the "non-repeat" logic.
    mcs_from_existing_sequence_for_priority_check = set().union(*[_get_assigned_mcs(d) for d in input_food_dicts]) if input_food_dicts else set()


    print(f"Priority Filter: Current sequence MCs for priority check: {mcs_from_existing_sequence_for_priority_check}")
    print(f"Raw results MCs for priority check: {[r.get('AssignedMC') for r in meal_type_results_raw]}")
    
    # Step 1: Try to find foods with priority MCs that are NOT already in the current sequence
    prioritized_results = []
    for result in meal_type_results_raw:
        assigned_mcs = {mc.strip() for mc in result.get("AssignedMC", "").split(",") if mc.strip()}
        
        # A food is prioritized if it has *any* priority MC AND its assigned MCs do NOT overlap
        # with MCs already present in the sequence (to avoid offering same priority MC again).
        if any(mc in priority_mc_list for mc in assigned_mcs) and \
           assigned_mcs.isdisjoint(mcs_from_existing_sequence_for_priority_check):
            prioritized_results.append(result)
    
    # Step 2: If prioritized results exist, return them
    if prioritized_results:
        print(f"Priority Filter: Found {len(prioritized_results)} results with priority MCs and no overlapping existing MCs.")
        print(f"Prioritized results MCs: {[r.get('AssignedMC') for r in prioritized_results]}")
        return prioritized_results
    
    # Step 3: If no priority MCs found (or all were already in sequence), return all original raw results.
    # The main exclusion filtering (like MC-MD non-repeat) is handled by _build_filter_clauses before this.
    print("Priority Filter: No new priority MCs found. Returning all raw results received.")
    return meal_type_results_raw # Return original results, as broader filtering is already done.

# Old version for scenario 3 AND 1 WORKING WELL
# def calculate_combined_metrics(food_dict_recommendation, total_nutrients_from_existing_sequence):
#     """
#     Calculates nutritional metrics for a *combination* of existing sequence nutrients
#     and the nutrients from the 'food_dict_recommendation'.
#     """
#     # Ensure food_dict_recommendation has serving_size, serving_g, and calorie data
#     # These should have been set by calculate_portion_value or similar logic before scoring.
#     rec_food_code = food_dict_recommendation.get('FoodCode', "N/A_comb_metrics")
#     try:
#         rec_serving_size = scoring_module.safe_float(food_dict_recommendation.get('serving_size'))
#         rec_serving_g = scoring_module.safe_float(food_dict_recommendation.get('serving_g'))

#         if rec_serving_size is None or rec_serving_g is None or rec_serving_size <= 0 or rec_serving_g <= 0:
#             print(f"Warn (calc_combined_metrics): Invalid serving data for recommendation {rec_food_code} (SS='{food_dict_recommendation.get('serving_size')}', SG='{food_dict_recommendation.get('serving_g')}'). Cannot calculate combined metrics.")
#             return None

#         # 1. Calculate actual nutrient values for the 'food_dict_recommendation'
#         nutrients_from_recommendation = {}
#         for key_internal, db_keys in nutrient_keys_map.items():
#             raw_val_per_100g_str = None
#             for db_key_option in db_keys:
#                 raw_val_per_100g_str = food_dict_recommendation.get(db_key_option)
#                 if raw_val_per_100g_str is not None:
#                     break
            
#             val_per_100g = scoring_module.safe_float(raw_val_per_100g_str, default=0.0) # Default to 0 if missing, to avoid None in calculations
            
#             # Actual nutrient value = (value_per_100g / 100) * serving_g * serving_size
#             actual_nutrient_value = (val_per_100g / 100.0) * rec_serving_g * rec_serving_size
#             nutrients_from_recommendation[key_internal] = actual_nutrient_value

#         # 2. Combine with nutrients from the existing sequence
#         combined_nutrients_total = total_nutrients_from_existing_sequence.copy() # Start with existing
#         for nutrient_key, rec_value in nutrients_from_recommendation.items():
#             combined_nutrients_total[nutrient_key] = combined_nutrients_total.get(nutrient_key, 0.0) + rec_value

#         # 3. Calculate final metrics based on these 'combined_nutrients_total'
#         # This part is similar to scoring_module.calculate_all_metrics_for_food,
#         # but operates on the already summed 'combined_nutrients_total'.
#         final_combined_metrics = {}
        
#         # Extract combined values, ensure no division by zero for denominators
#         c_energy = combined_nutrients_total.get('calories', 0.0)
#         c_energy_for_division = max(c_energy, 1.0) # Avoid division by zero if total energy is 0

#         c_protein = combined_nutrients_total.get('protein', 0.0)
#         c_protein_for_division = max(c_protein, 1.0)

#         c_carbs = combined_nutrients_total.get('carbohydrate', 0.0)
#         c_carbs_for_division = max(c_carbs, 1.0)

#         c_fiber = combined_nutrients_total.get('dietary_fiber', 0.0)
#         c_fiber_for_division = max(c_fiber, 1.0) # For carb_to_fiber

#         c_sugars = combined_nutrients_total.get('total_sugars', 0.0)
#         c_added_sugars = combined_nutrients_total.get('added_sugar_g') # Might be None

#         c_sat_fat = combined_nutrients_total.get('total_saturated', 0.0)
#         c_sodium = combined_nutrients_total.get('sodium', 0.0)

#         c_potassium = combined_nutrients_total.get('potassium', 0.0)
#         c_potassium_for_division = max(c_potassium, 1.0) # For sodium_to_potassium

#         c_mono_fat = combined_nutrients_total.get('total_monounsaturated', 0.0)
#         c_poly_fat = combined_nutrients_total.get('total_polyunsaturated', 0.0)

#         c_total_fat = combined_nutrients_total.get('total_fat', 0.0)
#         c_total_fat_for_division = max(c_total_fat, 1.0)

#         c_omega3 = combined_nutrients_total.get('omega_3') # Might be None
#         c_cholesterol = combined_nutrients_total.get('cholesterol', 0.0)
#         c_trans_fat = combined_nutrients_total.get('total_trans', 0.0)
#         c_gl = combined_nutrients_total.get('gl') # Might be None

#         # Calculate metrics
#         final_combined_metrics["protein_to_calories"] = c_protein / c_energy_for_division
#         final_combined_metrics["carb_to_fiber"] = c_carbs / c_fiber_for_division if c_fiber > 0 else float('inf') # Or a large number if c_fiber is 0
        
#         sugar_source_for_ratio = c_sugars # Default to total sugars
#         if c_added_sugars is not None: # Prefer added sugars if available
#             sugar_source_for_ratio = c_added_sugars
#         final_combined_metrics["added_sugars_to_calories"] = sugar_source_for_ratio / c_energy_for_division

#         final_combined_metrics["saturated_fat_to_calories"] = c_sat_fat / c_energy_for_division
#         final_combined_metrics["sodium_to_potassium"] = c_sodium / c_potassium_for_division if c_potassium > 0 else float('inf')
#         final_combined_metrics["fiber_per_100_calories"] = (c_fiber / c_energy_for_division) * 100
#         final_combined_metrics["total_fat_to_calories"] = c_total_fat / c_energy_for_division

#         if c_gl is not None:
#             final_combined_metrics["glycemic_load"] = c_gl # GL is usually absolute, not a ratio here

#         if c_omega3 is not None:
#             total_fatty_acids_for_omega_ratio = c_mono_fat + c_poly_fat + c_sat_fat
#             if total_fatty_acids_for_omega_ratio > 0:
#                 final_combined_metrics["omega3_to_fatty_acid"] = c_omega3 / total_fatty_acids_for_omega_ratio
#             else:
#                 final_combined_metrics["omega3_to_fatty_acid"] = float('inf') # Or 0, depending on how it's scored

#         final_combined_metrics["fiber_to_calories"] = c_fiber / c_energy_for_division # Redundant with fiber_per_100_calories but kept if used
#         final_combined_metrics["healthy_fats_to_total_fat"] = (c_poly_fat + c_mono_fat) / c_total_fat_for_division if c_total_fat > 0 else 0.0
#         final_combined_metrics["cholesterol_per_1000kcal"] = (c_cholesterol / c_energy_for_division) * 1000
#         final_combined_metrics["fiber_to_carb"] = c_fiber / c_carbs_for_division if c_carbs > 0 else float('inf')
#         final_combined_metrics["trans_fat_to_calories"] = c_trans_fat / c_energy_for_division
#         final_combined_metrics["carb_to_calories"] = c_carbs / c_energy_for_division
#         final_combined_metrics["healthy_fats_to_calories"] = (c_mono_fat + c_poly_fat) / c_energy_for_division
#         final_combined_metrics["carb_to_protein"] = c_carbs / c_protein_for_division if c_protein > 0 else float('inf')

#         # Important: Add the total combined calories as 'calorie_range' for scoring rules
#         final_combined_metrics['calorie_range'] = c_energy
#         # print(f"Debug (calc_combined_metrics): Food {rec_food_code}, Combined Energy: {c_energy:.1f}")

#         return final_combined_metrics

#     except Exception as e:
#         print(f"Error calculating combined metrics for recommendation {rec_food_code} when combined with sequence: {e}")
#         traceback.print_exc()
#         return None

def _assemble_final_recommendations(filtered_meal_type_results, paired_with_results):
    """
    Assembles final recommendation list.
    Prioritizes PAIRED_WITH results if they share an MC with filtered_meal_type_results.
    Then adds remaining unique filtered_meal_type_results.
    """
    print(f"\nStep 4: Assembling final list...")
    if not filtered_meal_type_results and not paired_with_results:
        print("Assemble: Both input lists (filtered_meal_type_results and paired_with_results) are empty.")
        return []

    # Determine the MCs present in the (already MC-priority-filtered) meal_type_results
    # These MCs guide which PAIRED_WITH items get priority.
    target_mcs_from_filtered_results = set().union(*[_get_assigned_mcs(food) for food in filtered_meal_type_results])
    print(f"Assemble: Target MCs from filtered_meal_type_results (e.g., MC-SA if that was selected): {target_mcs_from_filtered_results if target_mcs_from_filtered_results else '{}'}")

    final_results_dict = OrderedDict() # To maintain order and uniqueness by FoodCode
    priority1_count = 0 # PAIRED_WITH items that match target_mcs

    if target_mcs_from_filtered_results: # Only prioritize PAIRED_WITH if there are target MCs to match
        for food_pw in paired_with_results:
            food_code = food_pw.get('FoodCode')
            if food_code and food_code not in final_results_dict: # Ensure uniqueness
                # Check if the PAIRED_WITH food shares any MC with the target_mcs
                if not target_mcs_from_filtered_results.isdisjoint(_get_assigned_mcs(food_pw)):
                    final_results_dict[food_code] = food_pw
                    priority1_count += 1
    else:
        # If filtered_meal_type_results was empty (e.g., priority_return found nothing),
        # then there are no target MCs. We might still want to add PAIRED_WITH items if any.
        # For now, this path means PAIRED_WITH items won't be added if target_mcs is empty.
        # Consider if all PAIRED_WITH items should be added if filtered_meal_type_results is empty.
        # Based on current logic, if priority_return returned empty, this path won't add PAIRED_WITH.
        print("Assemble: No target MCs from filtered_meal_type_results, so PAIRED_WITH items matching these MCs won't be prioritized.")


    print(f"Assemble: Added {priority1_count} prioritized PAIRED_WITH results matching target MCs.")

    # Add remaining unique items from filtered_meal_type_results
    priority2_count = 0
    for food_mt_filtered in filtered_meal_type_results:
        food_code = food_mt_filtered.get('FoodCode')
        if food_code and food_code not in final_results_dict: # Ensure uniqueness
            final_results_dict[food_code] = food_mt_filtered
            priority2_count += 1
    print(f"Assemble: Added {priority2_count} unique items from filtered_meal_type_results.")

    # If target_mcs_from_filtered_results was empty, and we still want to include PAIRED_WITH items
    # that didn't get prioritized, they could be added here as a lower priority.
    # For example:
    # if not target_mcs_from_filtered_results: # if filtered_meal_type_results was empty or yielded no MCs
    #     non_priority_pw_count = 0
    #     for food_pw in paired_with_results:
    #         food_code = food_pw.get('FoodCode')
    #         if food_code and food_code not in final_results_dict:
    #             final_results_dict[food_code] = food_pw
    #             non_priority_pw_count +=1
    #         print(f"Assemble: Added {non_priority_pw_count} non-prioritized PAIRED_WITH results (as filtered_meal_type was empty).")


    final_prioritized_list = list(final_results_dict.values())
    print(f"Assemble: Final assembled list size before portion/scoring: {len(final_prioritized_list)}.")
    return final_prioritized_list

def _calculate_scen1_ss(food, target_cal_for_meal):
    """
    Calculates serving size for Scenario 1 (single item or first item context).
    Aims for the food item to contribute a reasonable portion of the target meal calories.
    Returns serving_size (0.5, 1.0, or 1.5) or None if calculation fails.
    """
    food_code = food.get('FoodCode', 'N/A_scen1_ss')
    try:
        cals_per_100g_str = food.get('Calories') or food.get('Energy')
        serving_g_str = food.get('serving_g')

        cals_per_100g = scoring_module.safe_float(cals_per_100g_str)
        serving_g = scoring_module.safe_float(serving_g_str)

        if serving_g is None or cals_per_100g is None or serving_g <= 0 or cals_per_100g < 0: # Calories can be 0
            reason = []
            if serving_g is None or serving_g <= 0: reason.append(f"invalid serving_g ('{serving_g_str}')")
            if cals_per_100g is None or cals_per_100g < 0: reason.append(f"invalid Calories/Energy ('{cals_per_100g_str}')")
            print(f"Warning (_calculate_scen1_ss): Cannot calculate SS for {food_code}. Reason: {', '.join(reason)}. Returning None.")
            return None
        if target_cal_for_meal is None or target_cal_for_meal <=0:
            print(f"Warning (_calculate_scen1_ss): Invalid target_cal_for_meal ({target_cal_for_meal}) for {food_code}. Returning default SS=1.0.")
            return 1.0 # Default if target calories are not sensible

        # Calories for one standard serving (serving_size = 1.0)
        calories_at_std_serving = (cals_per_100g / 100.0) * serving_g
        if calories_at_std_serving <= 0: # If food has 0 calories or invalid calculation
             print(f"Warning (_calculate_scen1_ss): Food {food_code} has ~0 calories at standard serving. Returning SS=1.0.")
             return 1.0 # Avoid division by zero or strange behavior for 0-calorie items

        # Define thresholds based on the target meal calories
        # If one standard serving is more than 50% (with 10% buffer) of target, suggest half serving
        upper_threshold = (0.50 * target_cal_for_meal) * 1.10
        # If one standard serving is less than 25% (with 10% buffer) of target, suggest 1.5 serving
        lower_threshold = (0.25 * target_cal_for_meal) * 0.90

        if calories_at_std_serving > upper_threshold:
            return 0.5
        elif calories_at_std_serving < lower_threshold:
            # Before returning 1.5, ensure 1.5 servings doesn't grossly overshoot.
            # For simplicity, we'll stick to the original logic for now.
            # A more complex check could be: if 1.5 * calories_at_std_serving > target_cal_for_meal * 0.75, then return 1.0 instead.
            return 1.5
        else:
            return 1.0
    except Exception as e:
        print(f"Error in _calculate_scen1_ss for {food_code}: {e}")
        traceback.print_exc()
        return None # Fallback

def _calculate_scen2_ss(food, target_calories_for_pair_item, calories_from_first_item):
    """
    Calculates serving size for the second item in a sequence (Scenario 2 context).
    Aims for the second item's calories to fill the gap to reach 'target_calories_for_pair_item'
    when combined with 'calories_from_first_item'.
    Returns serving_size (0.5 or 1.0) or a default if calculation fails.
    'target_calories_for_pair_item' is the target total for the first TWO items.
    """
    food_code = food.get('FoodCode', 'N/A_scen2_ss')
    try:
        serving_g_str = food.get('serving_g')
        cals_per_100g_str = food.get('Calories') or food.get('Energy')

        serving_g = scoring_module.safe_float(serving_g_str)
        cals_per_100g = scoring_module.safe_float(cals_per_100g_str)

        if serving_g is None or cals_per_100g is None or serving_g <= 0 or cals_per_100g < 0:
            print(f"Warn (_scen2_ss): Invalid nutritional data for {food_code} (sg='{serving_g_str}', cal_100g='{cals_per_100g_str}'). Defaulting SS=0.5")
            return 0.5 # Default to smaller serving if data is bad

        calories_at_std_serving = (cals_per_100g / 100.0) * serving_g
        if calories_at_std_serving <= 0:
            print(f"Warn (_scen2_ss): Standard calories for {food_code} is zero/negative. Defaulting SS=0.5")
            return 0.5

        # Calories needed from this recommended food to reach the target for the pair
        required_rec_calories = target_calories_for_pair_item - calories_from_first_item

        if required_rec_calories <= 0:
            # This means the first item already meets or exceeds the pair target.
            # We should still add a small portion of the second item if possible.
            print(f"Warn (_scen2_ss): First item ({calories_from_first_item:.1f} cal) already meets/exceeds pair target ({target_calories_for_pair_item:.1f} cal). Recommending SS=0.5 for {food_code}.")
            return 0.5 # Recommend a small portion

        # Ideal serving size to meet the 'required_rec_calories'
        ideal_serving_size_factor = required_rec_calories / calories_at_std_serving

        # Normalize to 0.5 or 1.0
        # If ideal_serving_size_factor suggests 0.75 of a standard serving or less, recommend 0.5 serving.
        # Otherwise, recommend a full standard serving (1.0).
        final_serving_size = 0.5 if ideal_serving_size_factor <= 0.75 else 1.0
        # print(f"Debug (_scen2_ss): {food_code} - req_cal: {required_rec_calories:.1f}, ideal_ss: {ideal_serving_size_factor:.2f}, final_ss: {final_serving_size}")
        return final_serving_size

    except Exception as e:
        print(f"Error in _calculate_scen2_ss for {food_code}: {e}")
        traceback.print_exc()
        return 0.5 # Fallback to a small serving size

def _get_food_actual_calories(food_dict, default_ss=None):
    """
    Calculates the actual calories for a food item based on its serving_size, serving_g, and Calories/100g.
    'default_ss' can be provided if food_dict might not have 'serving_size'.
    """
    food_code = food_dict.get('FoodCode', 'N/A_actual_cal')
    try:
        # Determine serving_size: use from food_dict, or default_ss, or fallback to None
        ss_str = food_dict.get('serving_size')
        serving_size = None
        if default_ss is not None:
            serving_size = scoring_module.safe_float(default_ss) # Use default if provided
        if serving_size is None and ss_str is not None : # If no default_ss or it was invalid, try from food_dict
            serving_size = scoring_module.safe_float(ss_str)

        serving_g_str = food_dict.get('serving_g')
        cals_per_100g_str = food_dict.get('Calories') or food_dict.get('Energy')

        serving_g = scoring_module.safe_float(serving_g_str)
        cals_per_100g = scoring_module.safe_float(cals_per_100g_str)

        # Validate all necessary components
        if serving_size is None or serving_g is None or cals_per_100g is None or \
           serving_size <= 0 or serving_g <= 0 or cals_per_100g < 0: # Calories can be 0
            reason = []
            if serving_size is None or serving_size <= 0: reason.append(f"invalid/missing serving_size ('{ss_str}' -> {serving_size}, default_ss: {default_ss})")
            if serving_g is None or serving_g <= 0: reason.append(f"invalid/missing serving_g ('{serving_g_str}')")
            if cals_per_100g is None or cals_per_100g < 0: reason.append(f"invalid/missing Calories/Energy ('{cals_per_100g_str}')")
            print(f"Error in _get_food_actual_calories for {food_code}: {', '.join(reason)}. Cannot calculate actual calories.")
            return None

        # Calories per defined standard serving (e.g., per 1 slice, per 1 cup if serving_g is that amount)
        # calories_per_standard_serving_unit = (cals_per_100g / 100.0) * serving_g
        # Actual calories = calories_per_standard_serving_unit * serving_size (the multiplier)
        actual_calories = (cals_per_100g / 100.0) * serving_g * serving_size
        return actual_calories

    except Exception as e:
        print(f"Error in _get_food_actual_calories for {food_code}: {e}")
        traceback.print_exc()
        return None


def calculate_portion_value(recommendations, target_calories_for_meal, input_foods):
    """
    Calculates and assigns 'serving_size' to each recommendation.
    Logic differs based on whether input_foods are present.
    This function MUTATES the food items in the 'recommendations' list.
    """
    print(f"\n--- Entering Portion Calculation ---")
    print(f"Target Calories for Meal: {target_calories_for_meal if target_calories_for_meal is not None else 'N/A'}")
    print(f"# Input foods: {len(input_foods)}")

    if not recommendations:
        print("Portion Calc: No recommendations to process.")
        return []

    if target_calories_for_meal is None or target_calories_for_meal <= 0:
        print("Portion Calc: Invalid or no target_calories_for_meal. Assigning None to serving_size for all recommendations.")
        for food in recommendations:
            food['serving_size'] = None # Or a default like 1.0, but None indicates it couldn't be calculated meaningfully
        return recommendations

    # Scenario 1: No input foods. Standard recommendations.
    if not input_foods:
        print("Portion Calc: Scenario - No input foods. Using _calculate_scen1_ss for each.")
        for food in recommendations:
            food['serving_size'] = _calculate_scen1_ss(food, target_calories_for_meal)
        # Filter out foods where serving_size could not be calculated
        recommendations = [food for food in recommendations if food.get('serving_size') is not None]

    # Scenario 2: One input food. Aim for a pair. (This logic is primarily for _execute_scenario_2 and _execute_scenario_3)
    # The calculate_portion_value function is more general.
    # If this function is called directly from _execute_scenario_1 with one input food,
    # it means we are trying to find companions.
    elif len(input_foods) == 1:
        print("Portion Calc: Scenario - One input food. Using _calculate_scen2_ss for recommendations.")
        input_food_item = input_foods[0]
        # We need the serving_size of the input_food_item to be already defined.
        # If it's called from test harness, it should be.
        calories_from_input_food = _get_food_actual_calories(input_food_item)

        if calories_from_input_food is not None:
            # The target for the *pair* (input_food + one_recommendation) is typically 0.75 of total meal target.
            # This 'target_calories_for_meal' is the full meal target.
            # Let's assume _calculate_scen2_ss expects the target for the *pair*.
            target_cal_for_the_pair = 0.75 * target_calories_for_meal # Example target for the pair
            print(f"Portion Calc: Input food calories: {calories_from_input_food:.1f}, Target for pair with new item: {target_cal_for_the_pair:.1f}")

            temp_recommendations = []
            for food_rec in recommendations:
                # _calculate_scen2_ss(food, target_calories_for_pair_item, calories_from_first_item)
                calculated_ss = _calculate_scen2_ss(food_rec, target_cal_for_the_pair, calories_from_input_food)
                food_rec['serving_size'] = calculated_ss
                if calculated_ss is not None:
                    # Further filter: does this make sense calorically?
                    # Actual calories of this rec
                    actual_cal_rec = _get_food_actual_calories(food_rec)
                    if actual_cal_rec is not None and actual_cal_rec > 0:
                        # Total calories if this rec is added
                        # total_pair_cal = calories_from_input_food + actual_cal_rec
                        # if total_pair_cal <= target_cal_for_pair * 1.15 and total_pair_cal >= target_cal_for_pair * 0.85: # within 15% range
                        temp_recommendations.append(food_rec)
                        # else:
                        #     print(f"Portion Calc: Rec {food_rec.get('FoodCode')} with SS {calculated_ss} results in pair cal {total_pair_cal}, outside range of {target_cal_for_pair}.")
                    # else:
                    #     print(f"Portion Calc: Rec {food_rec.get('FoodCode')} has no/invalid actual cals after SS calc. Removing.")
            recommendations = temp_recommendations
        else:
            print("Portion Calc: Could not get calories from the single input food. Applying _calculate_scen1_ss logic as fallback.")
            for food in recommendations: # Fallback to Scen1 logic
                food['serving_size'] = _calculate_scen1_ss(food, target_calories_for_meal)
            recommendations = [food for food in recommendations if food.get('serving_size') is not None]

    # Scenario "3" in your original calculate_portion_value: Multiple input foods (filter mode for Scen 1)
    # Here, recommendations are additional items. We give them SS=1.0 and check calorie limits.
    else: # len(input_foods) > 1
        print("Portion Calc: Scenario - Multiple input foods. Assigning SS=1.0 and filtering by total meal calories.")
        total_calories_from_input_foods = sum(filter(None, (_get_food_actual_calories(f) for f in input_foods)))
        if total_calories_from_input_foods is None: total_calories_from_input_foods = 0 # Should not happen if inputs are valid

        print(f"Portion Calc: Total calories from {len(input_foods)} input foods: {total_calories_from_input_foods:.1f}")
        filtered_recs_for_multi_input = []
        for food_rec in recommendations:
            # For new items to add to an existing list of inputs, assume SS=1.0 initially
            # The _get_food_actual_calories needs a serving_size. We use default_ss=1.0 for this check.
            calories_of_rec_at_ss1 = _get_food_actual_calories(food_rec, default_ss=1.0)

            if calories_of_rec_at_ss1 is not None and calories_of_rec_at_ss1 > 0:
                if (total_calories_from_input_foods + calories_of_rec_at_ss1) <= (target_calories_for_meal * 1.10): # Allow 10% overshoot for the total
                    food_rec['serving_size'] = 1.0 # Confirm SS=1.0
                    filtered_recs_for_multi_input.append(food_rec)
                # else:
                #     print(f"Portion Calc: Rec {food_rec.get('FoodCode')} (at SS=1.0, cal={calories_of_rec_at_ss1:.1f}) would exceed meal target with inputs. Removing.")
            # else:
            #     print(f"Portion Calc: Rec {food_rec.get('FoodCode')} has no/invalid actual cals at SS=1.0. Removing.")
        
        num_removed = len(recommendations) - len(filtered_recs_for_multi_input)
        if num_removed > 0:
            print(f"Portion Calc: Filtered out {num_removed} recommendations due to calorie limits with multiple inputs.")
        recommendations = filtered_recs_for_multi_input

    print(f"--- Exiting Portion Calculation ({len(recommendations)} recs with calculated SS) ---")
    return recommendations


def calculate_combined_metrics(food_dict_recommendation, total_nutrients_from_existing_sequence):
    """
    Calculates nutritional metrics for a *combination* of existing sequence nutrients
    and the nutrients from the 'food_dict_recommendation'.
    """
    # Ensure food_dict_recommendation has serving_size, serving_g, and calorie data
    # These should have been set by calculate_portion_value or similar logic before scoring.
    rec_food_code = food_dict_recommendation.get('FoodCode', "N/A_comb_metrics")
    try:
        rec_serving_size = scoring_module.safe_float(food_dict_recommendation.get('serving_size'))
        rec_serving_g = scoring_module.safe_float(food_dict_recommendation.get('serving_g'))

        if rec_serving_size is None or rec_serving_g is None or rec_serving_size <= 0 or rec_serving_g <= 0:
            print(f"Warn (calc_combined_metrics): Invalid serving data for recommendation {rec_food_code} (SS='{food_dict_recommendation.get('serving_size')}', SG='{food_dict_recommendation.get('serving_g')}'). Cannot calculate combined metrics.")
            return None

        # 1. Calculate actual nutrient values for the 'food_dict_recommendation'
        nutrients_from_recommendation = {}
        for key_internal, db_keys in nutrient_keys_map.items():
            raw_val_per_100g_str = None
            for db_key_option in db_keys:
                raw_val_per_100g_str = food_dict_recommendation.get(db_key_option)
                if raw_val_per_100g_str is not None:
                    break
            
            val_per_100g = scoring_module.safe_float(raw_val_per_100g_str, default=0.0) # Default to 0 if missing, to avoid None in calculations
            
            # Actual nutrient value = (value_per_100g / 100) * serving_g * serving_size
            actual_nutrient_value = (val_per_100g / 100.0) * rec_serving_g * rec_serving_size
            nutrients_from_recommendation[key_internal] = actual_nutrient_value

        # 2. Combine with nutrients from the existing sequence
        combined_nutrients_total = total_nutrients_from_existing_sequence.copy() # Start with existing
        for nutrient_key, rec_value in nutrients_from_recommendation.items():
            combined_nutrients_total[nutrient_key] = combined_nutrients_total.get(nutrient_key, 0.0) + rec_value

        # 3. Calculate final metrics based on these 'combined_nutrients_total'
        # This part is similar to scoring_module.calculate_all_metrics_for_food,
        # but operates on the already summed 'combined_nutrients_total'.
        final_combined_metrics = {}
        
        # Extract combined values, ensure no division by zero for denominators
        c_energy = combined_nutrients_total.get('calories', 0.0)
        c_energy_for_division = max(c_energy, 1.0) # Avoid division by zero if total energy is 0

        c_protein = combined_nutrients_total.get('protein', 0.0)
        c_protein_for_division = max(c_protein, 1.0)

        c_carbs = combined_nutrients_total.get('carbohydrate', 0.0)
        c_carbs_for_division = max(c_carbs, 1.0)

        c_fiber = combined_nutrients_total.get('dietary_fiber', 0.0)
        c_fiber_for_division = max(c_fiber, 1.0) # For carb_to_fiber

        c_sugars = combined_nutrients_total.get('total_sugars', 0.0)
        c_added_sugars = combined_nutrients_total.get('added_sugar_g') # Might be None

        c_sat_fat = combined_nutrients_total.get('total_saturated', 0.0)
        c_sodium = combined_nutrients_total.get('sodium', 0.0)

        c_potassium = combined_nutrients_total.get('potassium', 0.0)
        c_potassium_for_division = max(c_potassium, 1.0) # For sodium_to_potassium

        c_mono_fat = combined_nutrients_total.get('total_monounsaturated', 0.0)
        c_poly_fat = combined_nutrients_total.get('total_polyunsaturated', 0.0)

        c_total_fat = combined_nutrients_total.get('total_fat', 0.0)
        c_total_fat_for_division = max(c_total_fat, 1.0)

        c_omega3 = combined_nutrients_total.get('omega_3') # Might be None
        c_cholesterol = combined_nutrients_total.get('cholesterol', 0.0)
        c_trans_fat = combined_nutrients_total.get('total_trans', 0.0)
        c_gl = combined_nutrients_total.get('gl') # Might be None

        # Calculate metrics
        final_combined_metrics["protein_to_calories"] = c_protein / c_energy_for_division
        final_combined_metrics["carb_to_fiber"] = c_carbs / c_fiber_for_division if c_fiber > 0 else float('inf') # Or a large number if c_fiber is 0
        
        sugar_source_for_ratio = c_sugars # Default to total sugars
        if c_added_sugars is not None: # Prefer added sugars if available
            sugar_source_for_ratio = c_added_sugars
        final_combined_metrics["added_sugars_to_calories"] = sugar_source_for_ratio / c_energy_for_division

        final_combined_metrics["saturated_fat_to_calories"] = c_sat_fat / c_energy_for_division
        final_combined_metrics["sodium_to_potassium"] = c_sodium / c_potassium_for_division if c_potassium > 0 else float('inf')
        final_combined_metrics["fiber_per_100_calories"] = (c_fiber / c_energy_for_division) * 100
        final_combined_metrics["total_fat_to_calories"] = c_total_fat / c_energy_for_division

        if c_gl is not None:
            final_combined_metrics["glycemic_load"] = c_gl # GL is usually absolute, not a ratio here

        if c_omega3 is not None:
            total_fatty_acids_for_omega_ratio = c_mono_fat + c_poly_fat + c_sat_fat
            if total_fatty_acids_for_omega_ratio > 0:
                final_combined_metrics["omega3_to_fatty_acid"] = c_omega3 / total_fatty_acids_for_omega_ratio
            else:
                final_combined_metrics["omega3_to_fatty_acid"] = float('inf') # Or 0, depending on how it's scored

        final_combined_metrics["fiber_to_calories"] = c_fiber / c_energy_for_division # Redundant with fiber_per_100_calories but kept if used
        final_combined_metrics["healthy_fats_to_total_fat"] = (c_poly_fat + c_mono_fat) / c_total_fat_for_division if c_total_fat > 0 else 0.0
        final_combined_metrics["cholesterol_per_1000kcal"] = (c_cholesterol / c_energy_for_division) * 1000
        final_combined_metrics["fiber_to_carb"] = c_fiber / c_carbs_for_division if c_carbs > 0 else float('inf')
        final_combined_metrics["trans_fat_to_calories"] = c_trans_fat / c_energy_for_division
        final_combined_metrics["carb_to_calories"] = c_carbs / c_energy_for_division
        final_combined_metrics["healthy_fats_to_calories"] = (c_mono_fat + c_poly_fat) / c_energy_for_division
        final_combined_metrics["carb_to_protein"] = c_carbs / c_protein_for_division if c_protein > 0 else float('inf')

        # Important: Add the total combined calories as 'calorie_range' for scoring rules
        final_combined_metrics['calorie_range'] = c_energy
        # print(f"Debug (calc_combined_metrics): Food {rec_food_code}, Combined Energy: {c_energy:.1f}")

        return final_combined_metrics

    except Exception as e:
        print(f"Error calculating combined metrics for recommendation {rec_food_code} when combined with sequence: {e}")
        traceback.print_exc()
        return None

def calculate_scores_and_sort(recommendations_with_portions, input_foods_or_sequence, score_condition, meal_type):
    """
    Calculates scores for recommendations, normalizes them to 0-100 range, and sorts them.
    'input_foods_or_sequence' can be the initial list of input foods (for Scen 1)
    or the current sequence being built (for Scen 2 & 3).
    This function MUTATES the food items in 'recommendations_with_portions' by adding/updating 'score'.
    """
    print(f"\n--- Entering Score Calculation, Normalization & Sorting ---")
    print(f"Meal Type: {meal_type}, Score Condition: {score_condition}, #Input/Sequence items: {len(input_foods_or_sequence)}")

    if not recommendations_with_portions:
        print("Score Calc: No recommendations with portions to score.")
        return []

    meal_category_for_scoring = MEAL_TYPE_TO_SCORE_CATEGORY.get(meal_type, "main") # Default to "main"
    print(f"Score Calc: Using meal category '{meal_category_for_scoring}' for scoring rules.")

    # Pre-calculate total nutrients from input_foods_or_sequence if it's not empty
    total_nutrients_from_inputs = {key: 0.0 for key in nutrient_keys_map.keys()}
    # total_actual_calories_from_inputs = 0.0 # این متغیر در کد اصلی شما محاسبه میشد ولی مستقیما در منطق امتیازدهی این تابع استفاده نمیشد

    if input_foods_or_sequence:
        for i, item_in_input_or_seq in enumerate(input_foods_or_sequence):
            item_code = item_in_input_or_seq.get('FoodCode', f'InputItem_{i}')
            try:
                ss = scoring_module.safe_float(item_in_input_or_seq.get('serving_size'))
                sg = scoring_module.safe_float(item_in_input_or_seq.get('serving_g'))

                if ss is None or sg is None or ss <= 0 or sg <= 0:
                    # print(f"Warn (Score Calc): Invalid serving data for input/sequence item {item_code}. Skipping its nutrients.")
                    continue

                for key_internal, db_keys in nutrient_keys_map.items():
                    raw_val_per_100g_str = None
                    for db_key_option in db_keys:
                        raw_val_per_100g_str = item_in_input_or_seq.get(db_key_option)
                        if raw_val_per_100g_str is not None:
                            break
                    
                    val_per_100g = scoring_module.safe_float(raw_val_per_100g_str, 0.0)
                    actual_nutrient_val = (val_per_100g / 100.0) * sg * ss
                    total_nutrients_from_inputs[key_internal] += actual_nutrient_val
            except Exception as e:
                print(f"Error processing nutrients for input/sequence item {item_code}: {e}")
        
        # total_actual_calories_from_inputs = sum(filter(None, (_get_food_actual_calories(f) for f in input_foods_or_sequence)))
        # if total_actual_calories_from_inputs is None: total_actual_calories_from_inputs = 0.0

    # --- 1. Calculate Raw Scores ---
    for food_rec in recommendations_with_portions:
        food_rec_code = food_rec.get('FoodCode', 'N/A_rec_score')
        score = None # Initialize score as None, will be overwritten
        metrics_for_scoring_rules = None

        if input_foods_or_sequence:
            metrics_for_scoring_rules = calculate_combined_metrics(food_rec, total_nutrients_from_inputs)
        else:
            metrics_for_scoring_rules = scoring_module.calculate_all_metrics_for_food(food_rec, meal_category_for_scoring)
            if metrics_for_scoring_rules:
                actual_calories_of_rec = _get_food_actual_calories(food_rec)
                if actual_calories_of_rec is not None:
                    metrics_for_scoring_rules['calorie_range'] = actual_calories_of_rec
                elif 'calorie_range' in metrics_for_scoring_rules:
                    del metrics_for_scoring_rules['calorie_range']
        
        if metrics_for_scoring_rules:
            score = scoring_module.get_score_from_metrics(metrics_for_scoring_rules, score_condition, meal_category_for_scoring)
        
        # Assign the raw score (it might be None if calculation failed, though you mentioned no Nones)
        # If scoring_module guarantees a numerical score or a default for failure, this is fine.
        # For safety, let's assume score can still be None if metrics were not calculable.
        food_rec['score'] = score if score is not None else 0 # Default to 0 if score calculation failed

    # --- 2. Normalize Scores to 0-100 ---
    # Collect all (now numerical, or defaulted to 0) scores
    # Filter out items where score might still be None if the default 0 above is not used
    # or if scoring_module itself can return None and it's not caught
    
    # Assuming all 'score' are now numbers (either calculated or defaulted to 0)
    raw_scores = [rec['score'] for rec in recommendations_with_portions if rec.get('score') is not None]
    print('--------------------------------------------------------------------------')
    print('raw_scores: ', raw_scores)
    if not raw_scores: # No scores to normalize (e.g., all recommendations failed scoring)
        print("Score Norm: No valid raw scores to normalize.")
        # Sorting will happen on original (possibly all 0 or None) scores
    elif len(raw_scores) == 1 and len(recommendations_with_portions) == 1:
        # Only one item, conceptually it's the "best" among itself.
        # Or if only one item had a valid score
        if recommendations_with_portions[0].get('score') is not None:
             recommendations_with_portions[0]['score'] = 100.0
        print("Score Norm: Only one item with a score, set to 100.")
    else: # More than one score to normalize
        min_score = min(raw_scores)
        max_score = max(raw_scores)
        print(f"Score Norm: Raw scores range [{min_score}, {max_score}]")

        for food_rec in recommendations_with_portions:
            current_raw_score = food_rec.get('score')
            if current_raw_score is None: # Should not happen if defaulted to 0 above
                food_rec['score'] = 0 # Or handle as an error/log
                continue

            if max_score == min_score:
                # All valid raw scores are the same.
                # If that single score value is high (e.g. > 50 by some absolute standard), make them all 100.
                # If it's low, make them all, say, 10 or 0.
                # For simplicity here, if all are same, we set to 100 (meaning they are equally "best" among the current set).
                # A more nuanced approach: if min_score (which is also max_score) is e.g. 0, then normalized should be 0.
                # Let's use 100 if positive or zero, and 0 if negative (though scores are usually positive)
                if min_score > 0 : # if all scores are positive and equal
                     food_rec['score'] = 100.0
                elif min_score == 0: # if all scores are zero
                     food_rec['score'] = 0.0
                else: # if all scores are negative and equal (less common for scoring systems)
                     food_rec['score'] = 0.0 # Or some other representative value
                # Simpler: food_rec['score'] = 100.0 if min_score > 0 else (0.0 if min_score == 0 else 50.0) # 50 for negative same scores
            else:
                # Standard Min-Max normalization
                normalized_val = ((current_raw_score - min_score) / (max_score - min_score)) * 100.0
                food_rec['score'] = normalized_val
        print(f"Score Norm: Scores normalized to 0-100 range.")

    # --- 3. Sort by Normalized Score ---
    try:
        # Scores are now normalized (or original if normalization wasn't possible/meaningful)
        # Handle None scores defensively in sort key, though we try to avoid them
        sorted_recommendations = sorted(
            recommendations_with_portions,
            key=lambda x: x.get('score', -float('inf')) if x.get('score') is not None else -float('inf'),
            reverse=True
        )
        # print(f"Score Sort: Sorted {len(sorted_recommendations)} recommendations by (normalized) score.")
    except Exception as e:
        print(f"Score Sort: Error during sorting recommendations: {e}. Returning unsorted.")
        # traceback.print_exc() # Uncomment for debugging
        sorted_recommendations = recommendations_with_portions # Fallback

    print(f"--- Exiting Score Calculation, Normalization & Sorting ({len(sorted_recommendations)} recs) ---")
    return sorted_recommendations

def _normalize_candidate_scores(candidates_list):
    """
    Normalizes the 'score' of candidates in the list to a 0-100 range.
    Mutates the 'score' in the dictionaries within the list.
    """
    if not candidates_list:
        return []

    # Filter out candidates that might not have a score or have a None score
    # though the previous logic tries to ensure 'score' is present and not None.
    # For safety, we re-check here.
    scored_candidates = [cand for cand in candidates_list if cand.get('score') is not None]

    if not scored_candidates:
        # If no candidates have scores, return the original list (or handle as needed)
        # Scores might remain None or their original values.
        print("Score Norm (candidates): No valid raw scores to normalize among candidates.")
        return candidates_list # Or return an empty list if that's preferred when no scores

    raw_scores = [cand['score'] for cand in scored_candidates]

    if not raw_scores: # Should be redundant if scored_candidates is not empty
        print("Score Norm (candidates): Raw scores list is empty after filtering.")
        return candidates_list

    min_score = min(raw_scores)
    max_score = max(raw_scores)
    # print(f"Score Norm (candidates): Raw scores range [{min_score}, {max_score}]")

    # Apply normalization only to those candidates that had a score initially
    for candidate in scored_candidates: # Iterate only over candidates known to have a score
        current_raw_score = candidate['score'] # We know this exists and is not None

        if max_score == min_score:
            # All valid raw scores are the same.
            if min_score > 0:
                candidate['score'] = 100.0
            elif min_score == 0:
                candidate['score'] = 0.0
            else: # Negative scores, all same
                candidate['score'] = 0.0 # Or some other representative value like 50.0
        else:
            normalized_val = ((current_raw_score - min_score) / (max_score - min_score)) * 100.0
            candidate['score'] = normalized_val
    
    # For candidates that originally had no score (or None), their 'score' remains as it was.
    # The function returns the original list with scores of 'scored_candidates' mutated.
    return candidates_list
def _execute_scenario_1(driver, calorie_requirement, score_condition, meal_type,
                         input_food_dicts, required_conditions, required_allergens,
                         excluded_conditions, excluded_allergens, other_filters):
    """Implements Scenario 1: Standard recommendations."""
    food_dicts = input_food_dicts if input_food_dicts else []
    req_cond = required_conditions if required_conditions else []
    req_aller = required_allergens if required_allergens else []
    excl_cond = excluded_conditions if excluded_conditions else []
    excl_aller = excluded_allergens if excluded_allergens else []
    target_alias = "food"
    recommendation_list = None
    target_calories = None
    rounded_cr = round_cr(calorie_requirement)
    dist_key = MEAL_TYPE_TO_DIST_KEY.get(meal_type) if meal_type else None
    print(f"\nCalculating Target Calories: Input CR={calorie_requirement}, Rounded CR={rounded_cr}, Meal Key={dist_key}")
    if rounded_cr is not None and dist_key:
        print('Im in if :))))))))))))))))))))))))')
        meal_calories_dist = calorie_distributions.get(rounded_cr)
        if meal_calories_dist:
            target_calories = meal_calories_dist.get(dist_key)
        else:
            keys = sorted(calorie_distributions.keys())
            closest_cr = None
            if rounded_cr < keys[0]:
                closest_cr = keys[0]
            elif rounded_cr > keys[-1]:
                closest_cr = keys[-1]
            if closest_cr:
                print(f"Warning: CR {rounded_cr} out of range. Using boundary: {closest_cr}")
                target_calories = calorie_distributions[closest_cr].get(dist_key)
        if target_calories is None:
            print(f"Warning: Could not determine calories for CR {rounded_cr}/{dist_key}")
    else:
        print("Warning: Cannot calculate target calories.")
    if target_calories is not None:
        target_calories = float(target_calories)
    input_mc_list_for_exclusion = list(set().union(*[_get_assigned_mcs(d) for d in food_dicts]))
    if not food_dicts:
        print("\n--- Running Scenario 1: No Input Foods ---")
        if not meal_type or (meal_type not in MEAL_TYPE_TO_MO and meal_type != "SNACK"):
            print(f"Error: Valid meal_type required...")
            return None
        params = {}
        match_clause = f"MATCH ({target_alias}:Food)"
        base_where = []
        mo_target = MEAL_TYPE_TO_MO.get(meal_type)
        if meal_type == "SNACK":
            base_where.append(f"'{mo_target}' IN [val IN split(coalesce({target_alias}.AssignedMO, ''), ',') | trim(val)]")
        elif meal_type == "BREAKFAST":
            base_where.extend([
                f"'MO-BR' IN [val IN split(coalesce({target_alias}.AssignedMO, ''), ',') | trim(val)]",
                f"ANY(mc IN ['MC-MD', 'MC-SD'] WHERE mc IN [val IN split(coalesce({target_alias}.AssignedMC, ''), ',') | trim(val)])"
            ])
        elif mo_target:
            base_where.append(f"'{mo_target}' IN [val IN split(coalesce({target_alias}.AssignedMO, ''), ',') | trim(val)]")
        else:
            print(f"Error: Invalid meal type '{meal_type}'")
            return None
        filter_where = _build_filter_clauses(target_alias, params, req_cond, req_aller, excl_cond, excl_aller, other_filters)
        all_where = base_where + filter_where
        where_string = "WHERE " + " AND ".join(all_where) if all_where else ""
        query = f"{match_clause} {where_string} RETURN DISTINCT {target_alias}"
        print(f"Querying for {meal_type} by attributes...")
        print('_____________________________________________-')
        print(f"RAW DINNER Query: {query}")
        print(f"RAW DINNER Params: {params}")
        recommendation_list = _execute_recommendation_query(driver, query, params)
    else:
        start_code = food_dicts[-1].get('FoodCode')
        print(f"start_code: {start_code}, type: {type(start_code)}") # لاگ نوع FoodCode
        if start_code is not None:
            start_code = int(start_code) if isinstance(start_code, str) else start_code # تبدیل به Integer
        if not start_code:
            print(f"Error: No FoodCode in last input dict.")
            return None
        params_base = {"startFoodCodeParam": start_code}
        if meal_type == "SNACK":
            print("\n--- Running Scenario 1: Input Foods, Meal Type SNACK ---")
            params = {}
            match_clause = f"MATCH ({target_alias}:Food)"
            base_where = []
            mo_target = MEAL_TYPE_TO_MO.get(meal_type)
            if mo_target:
                base_where.append(f"'{mo_target}' IN [val IN split(coalesce({target_alias}.AssignedMO,''),',') | trim(val)]")
            else:
                print(f"Error: Invalid meal type '{meal_type}'")
                return None
            excluded_codes = [int(d.get('FoodCode')) for d in food_dicts if d.get('FoodCode')] # تبدیل به Integer
            if excluded_codes:
                params["excludedFoodCodes"] = excluded_codes
                base_where.append(f"NOT {target_alias}.FoodCode IN $excludedFoodCodes")
            filter_where = _build_filter_clauses(target_alias, params, req_cond, req_aller, excl_cond, excl_aller, other_filters)
            all_where = base_where + filter_where
            where_string = "WHERE " + " AND ".join(all_where) if all_where else ""
            query = f"{match_clause} {where_string} RETURN DISTINCT {target_alias}"
            print("Querying for SNACKs...")
            recommendation_list = _execute_recommendation_query(driver, query, params)
        elif meal_type == "BREAKFAST":
            print(f"\n--- Running Scenario 1: Input Foods, Meal Type BREAKFAST (Simple Flow) ---")
            params_br = params_base.copy()
            relationship_type = "BreakFast"
            match_clause_br = f"MATCH (startNode:Food {{FoodCode: $startFoodCodeParam}}) -[:{relationship_type}]- ({target_alias}:Food)"
            base_where_br = [f"startNode <> {target_alias}"]
            filter_where_br = _build_filter_clauses(target_alias, params_br, req_cond, req_aller, excl_cond, excl_aller, other_filters, input_mc_list_for_exclusion, "BREAKFAST")
            all_where_br = base_where_br + filter_where_br
            where_string_br = "WHERE " + " AND ".join(all_where_br) if all_where_br else ""
            query_br = f"{match_clause_br} {where_string_br} RETURN DISTINCT {target_alias}"
            print(f"Querying for {meal_type} rels...")
            recommendation_list = _execute_recommendation_query(driver, query_br, params_br)
        elif meal_type in COMPLEX_PRIORITY_MEALS:
            print(f"\n--- Running Scenario 1: Input Foods, Meal Type {meal_type} (Complex Flow) ---")
            params_mt = params_base.copy()
            relationship_type = meal_type
            match_clause_mt = f"MATCH (startNode:Food {{FoodCode: $startFoodCodeParam}}) -[:{relationship_type}]- ({target_alias}:Food)"
            base_where_mt = [f"startNode <> {target_alias}"]
            filter_where_mt = _build_filter_clauses(target_alias, params_mt, req_cond, req_aller, excl_cond, excl_aller, other_filters, input_mc_list_for_exclusion, meal_type)
            all_where_mt = base_where_mt + filter_where_mt
            where_string_mt = "WHERE " + " AND ".join(all_where_mt) if all_where_mt else ""
            query_mt = f"{match_clause_mt} {where_string_mt} RETURN DISTINCT {target_alias}"
            print(f"\nStep 1: Querying RAW {meal_type}...")
            meal_type_results_raw = _execute_recommendation_query(driver, query_mt, params_mt)
            print(f"Raw results: {[r.get('FoodCode') for r in meal_type_results_raw]}") # لاگ نتایج خام
            print(f"\nStep 2: Applying MC Priority Filter...")
            filtered_meal_type_results = priority_return(meal_type_results_raw, food_dicts)
            params_pw = params_base.copy()
            match_clause_pw = f"MATCH (startNode:Food {{FoodCode: $startFoodCodeParam}}) -[:PAIRED_WITH]- ({target_alias}:Food)"
            base_where_pw = [f"startNode <> {target_alias}"]
            filter_where_pw = _build_filter_clauses(target_alias, params_pw, req_cond, req_aller, excl_cond, excl_aller, other_filters, input_mc_list_for_exclusion, meal_type)
            all_where_pw = base_where_pw + filter_where_pw
            where_string_pw = "WHERE " + " AND ".join(all_where_pw) if all_where_pw else ""
            query_pw = f"{match_clause_pw} {where_string_pw} RETURN DISTINCT {target_alias}"
            print(f"\nStep 3: Querying PAIRED_WITH...")
            paired_with_results = _execute_recommendation_query(driver, query_pw, params_pw)
            recommendation_list = _assemble_final_recommendations(filtered_meal_type_results, paired_with_results)
        else:
            print(f"Error: Invalid or unsupported meal_type '{meal_type}'...")
            return None
    recommendations_with_portions = None
    if recommendation_list is not None:
        print(f"\nProceeding to portion calculation with {len(recommendation_list)} recommendations.")
        recommendations_with_portions = calculate_portion_value(recommendation_list, target_calories, food_dicts)
    else:
        print("No recommendations generated or error occurred, skipping portion & score calculation.")
        return None
    if recommendations_with_portions is not None:
        sorted_recommendations_with_score = calculate_scores_and_sort(
            recommendations_with_portions, food_dicts, score_condition, meal_type
        )
        return sorted_recommendations_with_score
    else:
        print("Error after portion calculation. Cannot calculate scores.")
        return None

################ OLD #############
def _execute_scenario_2(driver, calorie_requirement, score_condition, meal_type,
                         input_food_dicts, required_conditions, required_allergens,
                         excluded_conditions, excluded_allergens, other_filters):
    """Implements Scenario 2: Sequence Building with 3/4 rule for second item."""
    print("\n--- Running Scenario 2: Sequence Building (Modified 3/4 Rule) ---")
    if not input_food_dicts or len(input_food_dicts) != 1:
        print("Error (Scen 2): Requires 1 input food.")
        return None
    if not meal_type or meal_type == "SNACK":
        print(f"Error (Scen 2): Meal type '{meal_type}' not supported.")
        return None
    if meal_type not in POSSIBLE_MEAL_RELATIONSHIPS and meal_type != "PAIRED_WITH":
        print(f"Error: Invalid meal_type '{meal_type}' for relationship.")
        return None
    req_cond = required_conditions if required_conditions else []
    req_aller = required_allergens if required_allergens else []
    excl_cond = excluded_conditions if excluded_conditions else []
    excl_aller = excluded_allergens if excluded_allergens else []
    target_alias = "food"
    target_calories = None
    rounded_cr = round_cr(calorie_requirement)
    dist_key = MEAL_TYPE_TO_DIST_KEY.get(meal_type)
    print(f"Calculating Target Calories: Input CR={calorie_requirement}, Rounded CR={rounded_cr}, Meal Key={dist_key}")
    if rounded_cr is not None and dist_key:
        meal_calories_dist = calorie_distributions.get(rounded_cr)
        if meal_calories_dist:
            target_calories = meal_calories_dist.get(dist_key)
        else:
            keys = sorted(calorie_distributions.keys())
            closest_cr = None
            if rounded_cr < keys[0]:
                closest_cr = keys[0]
            elif rounded_cr > keys[-1]:
                closest_cr = keys[-1]
            if closest_cr:
                print(f"Warn: CR {rounded_cr} out of range. Using {closest_cr}")
                target_calories = calorie_distributions[closest_cr].get(dist_key)
        if target_calories is None:
            print(f"Warn: Could not get calories for {rounded_cr}/{dist_key}")
    if target_calories is None:
        print("Error (Scen 2): Could not determine target calories.")
        return None
    target_calories = float(target_calories)
    target_calories_for_pair = 0.75 * target_calories
    overall_target_with_tolerance = target_calories * 1.10
    print(f"Target Calories for Meal: {target_calories:.1f}, Target for Pair (Item 2 only): {target_calories_for_pair:.1f}")
    output_sequence = [input_food_dicts[0].copy()]
    current_food_node = output_sequence[0]
    current_total_calories = _get_food_actual_calories(current_food_node)
    if current_total_calories is None:
        print(f"Error (Scen 2): Cannot calculate calories for input food {current_food_node.get('FoodCode', 'N/A')}.")
        return None
    print(f"Initial Sequence: [{current_food_node.get('foodName', 'N/A')}] - Current Calories: {current_total_calories:.1f}")
    iteration_count = 0
    max_iterations = 10
    while current_total_calories < overall_target_with_tolerance and iteration_count < max_iterations:
        iteration_count += 1
        print(f"\n--- Sequence Iteration {iteration_count} ---")
        print(f"Current Node: {current_food_node.get('foodName', 'N/A')} ({current_food_node.get('FoodCode', 'N/A')})")
        print(f"Current Total Calories: {current_total_calories:.1f} / {target_calories:.1f}")
        start_code = current_food_node.get('FoodCode')
        if not start_code:
            print("Error: Lost FoodCode in sequence.")
            break
        params_base = {"startFoodCodeParam": start_code}
        current_sequence_mcs = list(set().union(*[_get_assigned_mcs(f) for f in output_sequence]))
        is_complex_meal = meal_type in COMPLEX_PRIORITY_MEALS
        params_mt = params_base.copy()
        relationship_type = "BreakFast" if meal_type == "BREAKFAST" else meal_type
        match_clause_mt = f"MATCH (startNode:Food {{FoodCode: $startFoodCodeParam}}) -[:{relationship_type}]- ({target_alias}:Food)"
        base_where_mt = [f"startNode <> {target_alias}"]
        filter_where_mt = _build_filter_clauses(target_alias, params_mt, req_cond, req_aller, excl_cond, excl_aller, other_filters, current_sequence_mcs, meal_type if is_complex_meal else None)
        all_where_mt = base_where_mt + filter_where_mt
        where_string_mt = "WHERE " + " AND ".join(all_where_mt) if all_where_mt else ""
        query_mt = f"{match_clause_mt} {where_string_mt} RETURN DISTINCT {target_alias}"
        print(f"Seq Build: Querying RAW {meal_type}...")
        meal_type_results_raw = _execute_recommendation_query(driver, query_mt, params_mt)
        filtered_meal_type_results = priority_return(meal_type_results_raw, output_sequence) if is_complex_meal else meal_type_results_raw
        params_pw = params_base.copy()
        match_clause_pw = f"MATCH (startNode:Food {{FoodCode: $startFoodCodeParam}}) -[:PAIRED_WITH]- ({target_alias}:Food)"
        base_where_pw = [f"startNode <> {target_alias}"]
        filter_where_pw = _build_filter_clauses(target_alias, params_pw, req_cond, req_aller, excl_cond, excl_aller, other_filters, current_sequence_mcs, meal_type if is_complex_meal else None)
        all_where_pw = base_where_pw + filter_where_pw
        where_string_pw = "WHERE " + " AND ".join(all_where_pw) if all_where_pw else ""
        query_pw = f"{match_clause_pw} {where_string_pw} RETURN DISTINCT {target_alias}"
        print(f"Seq Build: Querying PAIRED_WITH...")
        paired_with_results = _execute_recommendation_query(driver, query_pw, params_pw)
        potential_next_foods_structured = (
            _assemble_final_recommendations(filtered_meal_type_results, paired_with_results)
            if is_complex_meal
            else list(OrderedDict((f.get('FoodCode'), f) for f in paired_with_results + filtered_meal_type_results if f.get('FoodCode')).values())
        )
        print(f"Seq Build: Found {len(potential_next_foods_structured)} potential candidates.")
        if not potential_next_foods_structured:
            print("Seq Build: No more candidates. Stopping.")
            break
        candidates_with_details = []
        current_sequence_nutrients = {key: 0.0 for key in nutrient_keys_map.keys()}
        for food_in_seq in output_sequence:
            try:
                ss = scoring_module.safe_float(food_in_seq.get('serving_size'))
                sg = scoring_module.safe_float(food_in_seq.get('serving_g'))
                if ss is None or sg is None or ss <= 0 or sg <= 0:
                    print(f"Warn: Invalid serving data for {food_in_seq.get('FoodCode', 'N/A')}. Skipping nutrient calc.")
                    continue
                for key_internal, key_options in nutrient_keys_map.items():
                    raw_val_str = None
                    for option in key_options:
                        raw_val_str = food_in_seq.get(option)
                        if raw_val_str is not None:
                            break
                    raw_val_100g = scoring_module.safe_float(raw_val_str, 0.0)
                    actual_val = (raw_val_100g / 100.0) * sg * ss if raw_val_100g is not None else 0.0
                    current_sequence_nutrients[key_internal] += actual_val
            except Exception as e:
                print(f"Error processing nutrients for seq food {food_in_seq.get('FoodCode', 'N/A')}: {e}")
        meal_category = MEAL_TYPE_TO_SCORE_CATEGORY.get(meal_type, "main")
        for candidate in potential_next_foods_structured:
            candidate_copy = candidate.copy()
            initial_serving_size = _calculate_scen1_ss(candidate_copy, target_calories)
            if initial_serving_size is None:
                print(f"Warn: Skip candidate {candidate_copy.get('FoodCode', 'N/A')}: no initial SS.")
                continue
            candidate_copy['serving_size'] = initial_serving_size
            combined_metrics = calculate_combined_metrics(candidate_copy, current_sequence_nutrients)
            score = None
            actual_cand_cal = None
            if combined_metrics:
                actual_cand_cal = _get_food_actual_calories(candidate_copy)
                if actual_cand_cal is not None:
                    combined_metrics['calorie_range'] = current_sequence_nutrients.get('calories', 0.0) + actual_cand_cal
                elif 'calorie_range' in combined_metrics:
                    del combined_metrics['calorie_range']
                score = scoring_module.get_score_from_metrics(combined_metrics, score_condition, meal_category)
            else:
                print(f"Warn: Could not calc combined metrics for {candidate_copy.get('FoodCode', 'N/A')}")
            candidate_copy['score'] = score
            if score is not None:
                candidates_with_details.append(candidate_copy)
            else:
                print(f"Warn: Could not score candidate {candidate_copy.get('FoodCode', 'N/A')}. Skipping.")
        if not candidates_with_details:
            print("Seq Build: No candidates scored. Stopping.")
            break
        candidates_with_details = _normalize_candidate_scores(candidates_with_details)
        candidates_with_details.sort(key=lambda x: x.get('score', -float('inf')), reverse=True)
        print(f"Seq Build: Sorted {len(candidates_with_details)} candidates.")
        added_food_in_iteration = False
        for best_candidate in candidates_with_details:
            serving_size_to_check = best_candidate['serving_size']
            candidate_actual_calories = _get_food_actual_calories(best_candidate)
            if candidate_actual_calories is None or candidate_actual_calories <= 0:
                print(f"Warn: Candidate {best_candidate.get('FoodCode', 'N/A')} has invalid actual calories ({candidate_actual_calories}). Skipping.")
                continue
            print(f"Seq Build: Checking {best_candidate.get('foodName', 'N/A')} | Score:{best_candidate.get('score', -float('inf')):.2f} | SS:{serving_size_to_check} | Cal:{candidate_actual_calories:.1f}")
            if len(output_sequence) == 1:
                target_check = target_calories_for_pair * 1.10
                print(f"         (Check against Pair Target: {target_check:.1f})")
                if (current_total_calories + candidate_actual_calories) <= target_check:
                    print(f"         Adding {best_candidate.get('foodName', 'N/A')} with SS={serving_size_to_check}.")
                    output_sequence.append(best_candidate)
                    current_total_calories += candidate_actual_calories
                    current_food_node = best_candidate
                    added_food_in_iteration = True
                    break
                else:
                    if serving_size_to_check > 0.5:
                        print(f"         Trying SS=0.5...")
                        best_candidate_copy_half = best_candidate.copy()
                        best_candidate_copy_half['serving_size'] = 0.5
                        candidate_actual_calories_half = _get_food_actual_calories(best_candidate_copy_half)
                        if candidate_actual_calories_half is not None and \
                           (current_total_calories + candidate_actual_calories_half) <= target_check:
                            print(f"         Adding {best_candidate.get('foodName', 'N/A')} with SS=0.5 (New Cal: {candidate_actual_calories_half:.1f}).")
                            output_sequence.append(best_candidate_copy_half)
                            current_total_calories += candidate_actual_calories_half
                            current_food_node = best_candidate_copy_half
                            added_food_in_iteration = True
                            break
                        else:
                            print(f"         Still exceeds Pair Target even with SS=0.5. Skipping.")
                    else:
                        print(f"         Exceeds Pair Target and SS is already 0.5. Skipping.")
                    continue
            else:
                target_check = overall_target_with_tolerance
                print(f"         (Check against Overall Target: {target_check:.1f})")
                if (current_total_calories + candidate_actual_calories) <= target_check:
                    print(f"         Adding {best_candidate.get('foodName', 'N/A')} with SS={serving_size_to_check}.")
                    output_sequence.append(best_candidate)
                    current_total_calories += candidate_actual_calories
                    current_food_node = best_candidate
                    added_food_in_iteration = True
                    break
                else:
                    print(f"         Exceeds Overall Target ({target_check:.1f}). Skipping.")
                    continue
        if not added_food_in_iteration:
            print("Seq Build: No valid candidate found meeting constraints. Stopping.")
            break
    print(f"\n--- Sequence Building Finished ---")
    print(f"Final Sequence Length: {len(output_sequence)} items")
    print(f"Final Total Calories: {current_total_calories:.1f} / {target_calories:.1f}")
    return output_sequence

def _execute_scenario_3(driver, calorie_requirement, score_condition, meal_type,
                         input_food_dicts, # Should be empty for Scen 3
                         required_conditions, required_allergens,
                         excluded_conditions, excluded_allergens, other_filters, eaten_list):
    """
    Implements Scenario 3: Sequence Building with a Random First Food.
    - Selects a random first food item based on meal type and filters, excluding 'eaten_list'.
    - Builds a sequence of up to 4 items.
    - Uses _calculate_scen1_ss for the first item's serving size.
    - Uses _calculate_scen2_ss for the second item's serving size (aiming for pair target).
    - Uses _calculate_scen1_ss for subsequent items' serving sizes.
    - Applies calorie constraints at each step.
    - Scores and sorts candidates at each step with normalized scores in [0, 1].
    """
    print("\n--- Running Scenario 3: Sequence Building with Random First Food ---")
    print(f"Debug (Scen 3): Inputs: CR={calorie_requirement}, ScoreCond={score_condition}, MealType={meal_type}")
    print(f"Debug (Scen 3): Filters: ReqCond={required_conditions}, ReqAller={required_allergens}, ExclCond={excluded_conditions}, ExclAller={excluded_allergens}")
    print(f"Debug (Scen 3): OtherFilters={other_filters}, EatenList={eaten_list}")

    # --- 1. Input Validation & Initial Setup ---
    if input_food_dicts:
        print("Error (Scen 3): 'input_food_dicts' must be empty for Scenario 3. Aborting.")
        return []
    if not meal_type or meal_type == "SNACK":
        print(f"Error (Scen 3): Meal type '{meal_type}' is not supported for sequence building in Scenario 3. Aborting.")
        return []
    if meal_type not in POSSIBLE_MEAL_RELATIONSHIPS:
        print(f"Error (Scen 3): Invalid meal_type '{meal_type}' for relationship-based sequence building. Aborting.")
        return []

    req_cond = required_conditions if required_conditions else []
    req_aller = required_allergens if required_allergens else []
    excl_cond = excluded_conditions if excluded_conditions else []
    excl_aller = excluded_allergens if excluded_allergens else []
    eaten_food_codes = eaten_list if eaten_list else []

    target_node_alias = "food"

    # --- 2. Calculate Target Calories for the Meal ---
    rounded_cr = round_cr(calorie_requirement)
    meal_dist_key = MEAL_TYPE_TO_DIST_KEY.get(meal_type)
    print(f"Debug (Scen 3): Rounded CR={rounded_cr}, Meal Dist Key={meal_dist_key}")

    target_calories_for_meal = None
    if rounded_cr is not None and meal_dist_key:
        calorie_dist_for_cr = calorie_distributions.get(rounded_cr)
        if calorie_dist_for_cr:
            target_calories_for_meal = calorie_dist_for_cr.get(meal_dist_key)
        else:
            defined_cr_keys = sorted(calorie_distributions.keys())
            closest_cr_key = None
            if rounded_cr < defined_cr_keys[0]: closest_cr_key = defined_cr_keys[0]
            elif rounded_cr > defined_cr_keys[-1]: closest_cr_key = defined_cr_keys[-1]
            
            if closest_cr_key:
                print(f"Warn (Scen 3): CR {rounded_cr} is out of defined range. Using closest CR: {closest_cr_key} for calorie distribution.")
                target_calories_for_meal = calorie_distributions[closest_cr_key].get(meal_dist_key)
    
    if target_calories_for_meal is None:
        print(f"Error (Scen 3): Could not determine target calories for meal type '{meal_type}' with CR {calorie_requirement}. Aborting.")
        return []
    target_calories_for_meal = float(target_calories_for_meal)
    
    # Calculate pair target as 75% of meal target, with overall meal target being slightly flexible
    target_calories_for_first_pair = 0.75 * target_calories_for_meal
    overall_meal_target_with_tolerance = target_calories_for_meal * 1.10 # Allow 10% overshoot for total meal
    print(f"Debug (Scen 3): Target Calories -> Meal={target_calories_for_meal:.1f}, FirstPair={target_calories_for_first_pair:.1f}, OverallMax={overall_meal_target_with_tolerance:.1f}")

    # --- 3. Select a Random First Food ---
    # سعی می‌کنیم ابتدا یک غذای اصلی (MC-MD) تصادفی انتخاب کنیم
    main_meal_mc_for_first_item = "MC-MD" # شما می‌توانید این را به نیاز خود تغییر دهید
    
    params_for_first_food_main = {}
    main_where_conditions_first_food = []
    
    meal_occasion_code = MEAL_TYPE_TO_MO.get(meal_type)

    if meal_type == "BREAKFAST":
        main_where_conditions_first_food.extend([
            f"'{MEAL_TYPE_TO_MO['BREAKFAST']}' IN [val IN split(coalesce({target_node_alias}.AssignedMO, ''), ',') | trim(val)]",
            f"ANY(mc IN ['MC-MD', 'MC-SD'] WHERE mc IN [val IN split(coalesce({target_node_alias}.AssignedMC, ''), ',') | trim(val)])"
        ])
    elif meal_occasion_code:
        main_where_conditions_first_food.append(f"'{meal_occasion_code}' IN [val IN split(coalesce({target_node_alias}.AssignedMO, ''), ',') | trim(val)]")
    else:
        print(f"Error (Scen 3): Internal - meal_occasion_code not found for {meal_type}. Aborting.")
        return []

    # اضافه کردن فیلتر برای تضمین MC اصلی در غذای اول
    main_where_conditions_first_food.append(
        f"'{main_meal_mc_for_first_item}' IN [val IN split(coalesce({target_node_alias}.AssignedMC, ''), ',') | trim(val)]"
    )

    if eaten_food_codes:
        eaten_codes_str = [str(code) for code in eaten_food_codes]
        eaten_codes_int = [int(code) for code in eaten_food_codes if isinstance(code, (int, float)) or (isinstance(code, str) and code.isdigit())]
        params_for_first_food_main["excludedEatenFoodCodesStr"] = eaten_codes_str
        params_for_first_food_main["excludedEatenFoodCodesInt"] = eaten_codes_int
        main_where_conditions_first_food.append(f"NOT toString({target_node_alias}.FoodCode) IN $excludedEatenFoodCodesStr AND NOT toInteger({target_node_alias}.FoodCode) IN $excludedEatenFoodCodesInt")

    filter_clauses_first_food_main = _build_filter_clauses(
        target_node_alias, params_for_first_food_main,
        req_cond, req_aller, excl_cond, excl_aller, other_filters,
        input_mc_list_for_exclusion=None, # اینجا فیلتر MC برای اولین غذا اعمال نمیشه
        meal_type_for_mc_exclusion=None
    )
    
    all_where_clauses_first_food_main = main_where_conditions_first_food + filter_clauses_first_food_main
    where_string_first_food_main = "WHERE " + " AND ".join(all_where_clauses_first_food_main) if all_where_clauses_first_food_main else ""
    
    match_clause_first_food = f"MATCH ({target_node_alias}:Food)" 
    query_first_food_main = f"{match_clause_first_food} {where_string_first_food_main} RETURN DISTINCT {target_node_alias}"
    
    print(f"Debug (Scen 3): Trying to select main food ({main_meal_mc_for_first_item}) as first item.")
    print(f"Debug (Scen 3): Main Food Query: {query_first_food_main}")
    print(f"Debug (Scen 3): Main Food Params: {params_for_first_food_main}")
    potential_first_foods = _execute_recommendation_query(driver, query_first_food_main, params_for_first_food_main)

    # Fallback اگر غذای اصلی پیدا نشد: یک غذای عمومی‌تر برای وعده غذایی پیدا کن
    if not potential_first_foods:
        print(f"Warn (Scen 3): No main meal food found with MC '{main_meal_mc_for_first_item}'. Falling back to general meal items.")
        
        fallback_params_for_first_food = {}
        fallback_where_conditions_first_food = []
        
        # Meal Occasion (MO) filter remains
        if meal_occasion_code:
            fallback_where_conditions_first_food.append(f"'{meal_occasion_code}' IN [val IN split(coalesce({target_node_alias}.AssignedMO, ''), ',') | trim(val)]")
        
        # Excluded foods (eaten_list) remain
        if eaten_food_codes:
            eaten_codes_str = [str(code) for code in eaten_food_codes]
            eaten_codes_int = [int(code) for code in eaten_food_codes if isinstance(code, (int, float)) or (isinstance(code, str) and code.isdigit())]
            fallback_params_for_first_food["excludedEatenFoodCodesStr"] = eaten_codes_str
            fallback_params_for_first_food["excludedEatenFoodCodesInt"] = eaten_codes_int
            fallback_where_conditions_first_food.append(f"NOT toString({target_node_alias}.FoodCode) IN $excludedEatenFoodCodesStr AND NOT toInteger({target_node_alias}.FoodCode) IN $excludedEatenFoodCodesInt")

        # Other filters remain
        fallback_filter_clauses_first_food = _build_filter_clauses(
            target_node_alias, fallback_params_for_first_food,
            req_cond, req_aller, excl_cond, excl_aller, other_filters,
            input_mc_list_for_exclusion=None,
            meal_type_for_mc_exclusion=None
        )

        all_where_clauses_fallback = fallback_where_conditions_first_food + fallback_filter_clauses_first_food
        where_string_fallback = "WHERE " + " AND ".join(all_where_clauses_fallback) if all_where_clauses_fallback else ""
        query_first_food_fallback = f"{match_clause_first_food} {where_string_fallback} RETURN DISTINCT {target_node_alias}"

        print(f"Debug (Scen 3): Fallback First Food Query: {query_first_food_fallback}")
        print(f"Debug (Scen 3): Fallback First Food Params: {fallback_params_for_first_food}")
        potential_first_foods = _execute_recommendation_query(driver, query_first_food_fallback, fallback_params_for_first_food)

    if not potential_first_foods:
        print("Error (Scen 3): No valid first food items found matching any criteria. Aborting.")
        return []

    random_first_food_raw = random.choice(potential_first_foods)
    
    initial_ss_for_first_item = _calculate_scen1_ss(random_first_food_raw, target_calories_for_meal)
    
    if initial_ss_for_first_item is None:
        print(f"Error (Scen 3): Could not calculate initial serving size for the randomly selected first food ({random_first_food_raw.get('FoodCode', 'N/A')}). Aborting.")
        return []
    random_first_food_raw['serving_size'] = initial_ss_for_first_item
    
    # Debug log for the first food selected
    print(f"Debug (Scen 3): First Food Selected: {random_first_food_raw.get('foodName', 'N/A')} (Code: {random_first_food_raw.get('FoodCode')})")
    print(f"Debug (Scen 3):   SS={random_first_food_raw.get('serving_size')}, Calculated Cal: {_get_food_actual_calories(random_first_food_raw):.1f}")
    
    output_sequence = [random_first_food_raw.copy()]
    current_food_node_for_rels = output_sequence[0]

    current_total_calories_in_sequence = _get_food_actual_calories(current_food_node_for_rels)
    
    if current_total_calories_in_sequence is None:
        print(f"Error (Scen 3): Could not calculate actual calories for the chosen first food ({current_food_node_for_rels.get('FoodCode', 'N/A')}). Aborting.")
        return []

    print(f"Debug (Scen 3): Initial Sequence: [{current_food_node_for_rels.get('foodName', 'N/A')} ({current_food_node_for_rels.get('FoodCode')})] - Calories: {current_total_calories_in_sequence:.1f}, SS: {current_food_node_for_rels.get('serving_size')}")

    # --- 4. Iteratively Build the Rest of the Sequence (up to 4 items total) ---
    MAX_ITEMS_IN_SEQUENCE = 4
    sequence_iteration_count = 0
    MAX_ADD_ATTEMPTS = 10 # To prevent infinite loops if no suitable food is found

    while len(output_sequence) < MAX_ITEMS_IN_SEQUENCE and \
          current_total_calories_in_sequence < overall_meal_target_with_tolerance and \
          sequence_iteration_count < MAX_ADD_ATTEMPTS:
        
        sequence_iteration_count += 1
        print(f"\n--- Sequence Iteration {sequence_iteration_count} (Scen 3) ---")
        print(f"Debug (Scen 3) - Current state: Sequence Length={len(output_sequence)}, Total Calories={current_total_calories_in_sequence:.1f} / Overall Meal Target={overall_meal_target_with_tolerance:.1f}")
        print(f"Debug (Scen 3) - Finding next item related to: {current_food_node_for_rels.get('foodName', 'N/A')} (Code: {current_food_node_for_rels.get('FoodCode')})")

        start_node_food_code = current_food_node_for_rels.get('FoodCode')
        if not start_node_food_code:
            print("Error (Scen 3): Lost FoodCode in sequence iteration. Stopping.")
            break

        params_for_next_item_query = {"startFoodCodeParam": start_node_food_code}
        
        # گام ۲.۱: تشخیص MCهای ضروری و موجود در دنباله فعلی
        mcs_from_current_sequence = set().union(*[_get_assigned_mcs(f) for f in output_sequence])
        
        # MCهای اصلی که نمی‌خواهیم تکرار شوند
        # MC-MD: Main Dish, MC-SA: Salad/Appetizer (if considered primary), MC-BV: Beverage
        # این لیست رو بر اساس معنای MCهای دیتابیس خودت تنظیم کن.
        # مثلاً اگر MC-AP و MC-SD رو می‌خوای حتماً داشته باشی، اینجا فقط MC-MD رو به عنوان "اصلی" و "تکرار نشو" بزار.
        main_meal_non_repeat_mcs = {"MC-MD"} # مثلاً فقط غذای اصلی تکرار نشه.
        
        # MCهایی که باید از کوئری حذف بشن (فقط MCهای اصلی که قبلاً در دنباله وجود دارن)
        mcs_to_exclude_in_query = [mc for mc in mcs_from_current_sequence if mc in main_meal_non_repeat_mcs]
        
        # گام ۲.۲: تعیین MC هدف برای جستجوی آیتم بعدی
        target_mc_for_next_item = None
        
        # اگر هنوز AP نداریم و تعداد آیتم‌ها کمتر از MAX_ITEMS_IN_SEQUENCE هست
        if "MC-AP" not in mcs_from_current_sequence and len(output_sequence) < MAX_ITEMS_IN_SEQUENCE:
            target_mc_for_next_item = "MC-AP"
            print(f"Debug (Scen 3): Prioritizing search for 'MC-AP' for next item.")
        # اگر هنوز SD نداریم و AP رو یا داریم یا به دنبالش نیستیم و تعداد آیتم‌ها کمتر از MAX_ITEMS_IN_SEQUENCE هست
        elif "MC-SD" not in mcs_from_current_sequence and len(output_sequence) < MAX_ITEMS_IN_SEQUENCE:
            target_mc_for_next_item = "MC-SD"
            print(f"Debug (Scen 3): Prioritizing search for 'MC-SD' for next item.")
        else:
            print(f"Debug (Scen 3): AP and SD (or max items) criteria met. Searching for general related items.")

        # گام ۲.۳: ساخت کوئری‌های Neo4j با منطق فیلتر جدید
        is_complex_meal_type = meal_type in COMPLEX_PRIORITY_MEALS
        meal_relationship_type = "BreakFast" if meal_type == "BREAKFAST" else meal_type
        
        # --- برای Meal Relationship Query (مثلاً DINNER یا BREAKFAST) ---
        match_clause_meal_rels = f"MATCH (startNode:Food {{FoodCode: $startFoodCodeParam}}) -[:{meal_relationship_type}]- ({target_node_alias}:Food)"
        
        base_where_meal_rels = [f"startNode <> {target_node_alias}"]
        
        # حذف FoodCodeهای موجود در دنباله
        current_sequence_food_codes = [f.get('FoodCode') for f in output_sequence if f.get('FoodCode')]
        if current_sequence_food_codes:
            params_for_next_item_query["currentSeqFoodCodes_mt"] = current_sequence_food_codes
            base_where_meal_rels.append(f"NOT {target_node_alias}.FoodCode IN $currentSeqFoodCodes_mt")

        # اضافه کردن فیلتر هدفمند برای MC (AP یا SD)
        if target_mc_for_next_item:
            base_where_meal_rels.append(
                f"'{target_mc_for_next_item}' IN [val IN split(coalesce({target_node_alias}.AssignedMC, ''), ',') | trim(val)]"
            )

        # ساخت لیست MCهایی که باید از کوئری حذف بشن (فقط اونایی که در main_meal_non_repeat_mcs هستند)
        input_mc_list_for_exclusion_for_filter = mcs_to_exclude_in_query if is_complex_meal_type and mcs_to_exclude_in_query else None

        filters_meal_rels = _build_filter_clauses(
            target_node_alias, params_for_next_item_query,
            req_cond, req_aller, excl_cond, excl_aller, other_filters,
            input_mc_list_for_exclusion=input_mc_list_for_exclusion_for_filter,
            meal_type_for_mc_exclusion=meal_type if is_complex_meal_type else None
        )
        all_where_meal_rels = base_where_meal_rels + filters_meal_rels
        where_string_meal_rels = "WHERE " + " AND ".join(all_where_meal_rels) if all_where_meal_rels else ""
        query_meal_rels = f"{match_clause_meal_rels} {where_string_meal_rels} RETURN DISTINCT {target_node_alias}"

        # --- برای Paired With Query ---
        params_for_paired_with_query = {"startFoodCodeParam": start_node_food_code}

        match_clause_paired_with = f"MATCH (startNode:Food {{FoodCode: $startFoodCodeParam}}) -[:PAIRED_WITH]- ({target_node_alias}:Food)"
        base_where_paired_with = [f"startNode <> {target_node_alias}"]
        if current_sequence_food_codes:
            params_for_paired_with_query["currentSeqFoodCodes_pw"] = current_sequence_food_codes
            base_where_paired_with.append(f"NOT {target_node_alias}.FoodCode IN $currentSeqFoodCodes_pw")

        # اضافه کردن فیلتر هدفمند برای MC (AP یا SD)
        if target_mc_for_next_item:
            base_where_paired_with.append(
                f"'{target_mc_for_next_item}' IN [val IN split(coalesce({target_node_alias}.AssignedMC, ''), ',') | trim(val)]"
            )

        filters_paired_with = _build_filter_clauses(
            target_node_alias, params_for_paired_with_query,
            req_cond, req_aller, excl_cond, excl_aller, other_filters,
            input_mc_list_for_exclusion=input_mc_list_for_exclusion_for_filter,
            meal_type_for_mc_exclusion=meal_type if is_complex_meal_type else None
        )
        all_where_paired_with = base_where_paired_with + filters_paired_with
        where_string_paired_with = "WHERE " + " AND ".join(all_where_paired_with) if all_where_paired_with else ""
        query_paired_with = f"{match_clause_paired_with} {where_string_paired_with} RETURN DISTINCT {target_node_alias}"

        print(f"Debug (Scen 3): Meal Rel Query: {query_meal_rels}")
        print(f"Debug (Scen 3): Meal Rel Params: {params_for_next_item_query}")
        raw_meal_type_candidates = _execute_recommendation_query(driver, query_meal_rels, params_for_next_item_query)
        print(f"Debug (Scen 3): Found {len(raw_meal_type_candidates)} raw meal type candidates.")

        filtered_meal_type_candidates = []
        if is_complex_meal_type:
            filtered_meal_type_candidates = priority_return(raw_meal_type_candidates, output_sequence)
        else:
            filtered_meal_type_candidates = raw_meal_type_candidates
        print(f"Debug (Scen 3): After priority_return, {len(filtered_meal_type_candidates)} meal type candidates remain.")

        print(f"Debug (Scen 3): Paired With Query: {query_paired_with}")
        print(f"Debug (Scen 3): Paired With Params: {params_for_paired_with_query}")
        paired_with_candidates = _execute_recommendation_query(driver, query_paired_with, params_for_paired_with_query)
        print(f"Debug (Scen 3): Found {len(paired_with_candidates)} paired with candidates.")

        potential_next_foods_structured = []
        if is_complex_meal_type:
            potential_next_foods_structured = _assemble_final_recommendations(filtered_meal_type_candidates, paired_with_candidates)
        else:
            combined_candidates_simple = OrderedDict()
            for food in paired_with_candidates + filtered_meal_type_candidates:
                if food.get('FoodCode') and food.get('FoodCode') not in combined_candidates_simple:
                    combined_candidates_simple[food.get('FoodCode')] = food
            potential_next_foods_structured = list(combined_candidates_simple.values())
        
        print(f"Debug (Scen 3): Total {len(potential_next_foods_structured)} unique candidates after assembly/combination.")

        # --- START OF FALLBACK LOGIC INSERTION ---
        if not potential_next_foods_structured and target_mc_for_next_item:
            print(f"Debug (Scen 3): No candidates found for prioritized MC '{target_mc_for_next_item}' via relationships. Trying general search for related items (without specific MC target).")
            
            # --- ساخت پارامترهای جدید و تمیز برای کوئری‌های Fallback ---
            fallback_params = {} 

            # **NEW LINE HERE**: Add startFoodCodeParam to fallback_params
            fallback_params["startFoodCodeParam"] = start_node_food_code 

            # FoodCodeهای موجود در دنباله (برای exclude کردن)
            fallback_current_sequence_food_codes_int = [int(f.get('FoodCode')) for f in output_sequence if f.get('FoodCode') is not None]
            if fallback_current_sequence_food_codes_int:
                fallback_params["fallbackCurrentSeqFoodCodes"] = fallback_current_sequence_food_codes_int
          
            if input_mc_list_for_exclusion_for_filter:
                pass # This is correctly handled by _build_filter_clauses below
            
            fallback_base_where_meal_rels = [f"startNode <> {target_node_alias}"]
            if fallback_current_sequence_food_codes_int:
                # Use the new parameter name for fallback: $fallbackCurrentSeqFoodCodes
                fallback_base_where_meal_rels.append(f"NOT {target_node_alias}.FoodCode IN $fallbackCurrentSeqFoodCodes")

            # فراخوانی _build_filter_clauses با پارامترهای جدید Fallback
            fallback_filters_meal_rels = _build_filter_clauses(
                target_node_alias, fallback_params, # Make sure fallback_params is passed here
                req_cond, req_aller, excl_cond, excl_aller, other_filters,
                input_mc_list_for_exclusion=input_mc_list_for_exclusion_for_filter, # Pass this as argument
                meal_type_for_mc_exclusion=meal_type if is_complex_meal_type else None
            )
            fallback_all_where_meal_rels = fallback_base_where_meal_rels + fallback_filters_meal_rels
            fallback_where_string_meal_rels = "WHERE " + " AND ".join(fallback_all_where_meal_rels) if fallback_all_where_meal_rels else ""
            fallback_query_meal_rels = f"{match_clause_meal_rels} {fallback_where_string_meal_rels} RETURN DISTINCT {target_node_alias}"
            
            # --- بازسازی base_where clauses برای کوئری Paired With (بدون target_mc_for_next_item) ---
            fallback_base_where_paired_with = [f"startNode <> {target_node_alias}"]
            if fallback_current_sequence_food_codes_int:
                # Use the new parameter name for fallback: $fallbackCurrentSeqFoodCodes
                fallback_base_where_paired_with.append(f"NOT {target_node_alias}.FoodCode IN $fallbackCurrentSeqFoodCodes")

            fallback_filters_paired_with = _build_filter_clauses(
                target_node_alias, fallback_params, # Make sure fallback_params is passed here
                req_cond, req_aller, excl_cond, excl_aller, other_filters,
                input_mc_list_for_exclusion=input_mc_list_for_exclusion_for_filter, # Pass this as argument
                meal_type_for_mc_exclusion=meal_type if is_complex_meal_type else None
            )
            fallback_all_where_paired_with = fallback_base_where_paired_with + fallback_filters_paired_with
            fallback_where_string_paired_with = "WHERE " + " AND ".join(fallback_all_where_paired_with) if fallback_all_where_paired_with else ""
            fallback_query_paired_with = f"{match_clause_paired_with} {fallback_where_string_paired_with} RETURN DISTINCT {target_node_alias}"

            # Execute the fallback queries with the correctly populated fallback_params
            print(f"Debug (Scen 3): Fallback Meal Rel Query: {fallback_query_meal_rels}")
            print(f"Debug (Scen 3): Fallback Meal Rel Params: {fallback_params}") 
            fallback_meal_type_candidates = _execute_recommendation_query(driver, fallback_query_meal_rels, fallback_params)
            
            print(f"Debug (Scen 3): Fallback Paired With Query: {fallback_query_paired_with}")
            print(f"Debug (Scen 3): Fallback Paired With Params: {fallback_params}") 
            fallback_paired_with_candidates = _execute_recommendation_query(driver, fallback_query_paired_with, fallback_params)

            if is_complex_meal_type:
                potential_next_foods_structured = _assemble_final_recommendations(fallback_meal_type_candidates, fallback_paired_with_candidates)
            else:
                combined_candidates_simple = OrderedDict()
                for food in fallback_paired_with_candidates + fallback_meal_type_candidates:
                    if food.get('FoodCode') and food.get('FoodCode') not in combined_candidates_simple:
                        combined_candidates_simple[food.get('FoodCode')] = food
                potential_next_foods_structured = list(combined_candidates_simple.values())
            
            print(f"Debug (Scen 3): Found {len(potential_next_foods_structured)} candidates after general search fallback (without specific MC target).")

            if not potential_next_foods_structured:
                print("Seq Build (Scen 3): No candidates found even with general search fallback. Stopping sequence.")
                break
        # --- END OF FALLBACK LOGIC INSERTION ---

        if not potential_next_foods_structured:
            print("Seq Build (Scen 3): No more potential candidates found from relationships. Stopping sequence.")
            break

        current_sequence_total_nutrients = {key: 0.0 for key in nutrient_keys_map.keys()}
        # Ensure that current_total_calories_in_sequence is accurately derived from current_sequence_total_nutrients['calories']
        # after summing up all items in output_sequence for nutrient calculations
        for item_in_seq in output_sequence:
            ss_seq = scoring_module.safe_float(item_in_seq.get('serving_size'))
            sg_seq = scoring_module.safe_float(item_in_seq.get('serving_g'))
            if ss_seq is None or sg_seq is None or ss_seq <= 0 or sg_seq <= 0:
                print(f"Debug (Scen 3): Warn: Invalid serving data for seq item {item_in_seq.get('FoodCode', 'N/A')}. Skipping nutrient calc.")
                continue
            for key_internal, db_keys in nutrient_keys_map.items():
                raw_val_str = None
                for option in db_keys:
                    raw_val_str = item_in_seq.get(option)
                    if raw_val_str is not None: break
                val_100g = scoring_module.safe_float(raw_val_str, 0.0)
                actual_val = (val_100g / 100.0) * sg_seq * ss_seq
                current_sequence_total_nutrients[key_internal] += actual_val

        current_total_calories_in_sequence = current_sequence_total_nutrients.get('calories', 0.0)
        print(f"Debug (Scen 3): Re-calculated current_total_calories_in_sequence from combined nutrients for scoring: {current_total_calories_in_sequence:.1f}")


        meal_category_for_scoring = MEAL_TYPE_TO_SCORE_CATEGORY.get(meal_type, "main")
        
        candidates_with_ss_and_score = []
        for candidate_food in potential_next_foods_structured:
            candidate_copy = candidate_food.copy()
            
            calculated_ss_for_this_candidate = None
            if len(output_sequence) == 1: # This is for the second item in the sequence
                calculated_ss_for_this_candidate = _calculate_scen2_ss(
                    candidate_copy,
                    target_calories_for_first_pair, # This is the target for the pair
                    current_total_calories_in_sequence # This is the calories from the first item
                )
                # print(f"Debug (Scen 3): _calculate_scen2_ss for {candidate_copy.get('FoodCode', 'N/A')} -> TargetPair={target_calories_for_first_pair:.1f}, CurrentSeqCal={current_total_calories_in_sequence:.1f}, ResultSS={calculated_ss_for_this_candidate}")

            else: # This is for the third and fourth items
                calculated_ss_for_this_candidate = _calculate_scen1_ss(candidate_copy, target_calories_for_meal)
                # print(f"Debug (Scen 3): _calculate_scen1_ss for {candidate_copy.get('FoodCode', 'N/A')} -> TargetMeal={target_calories_for_meal:.1f}, ResultSS={calculated_ss_for_this_candidate}")
            
            if calculated_ss_for_this_candidate is None:
                print(f"Debug (Scen 3): Skipping {candidate_copy.get('FoodCode', 'N/A')} due to None serving_size from portion calculation.")
                continue
            candidate_copy['serving_size'] = calculated_ss_for_this_candidate
            
            # Recalculate combined nutrients and score with the assigned serving_size
            combined_metrics = calculate_combined_metrics(candidate_copy, current_sequence_total_nutrients)
            
            score_for_this_candidate = None
            if combined_metrics:
                score_for_this_candidate = scoring_module.get_score_from_metrics(
                    combined_metrics,
                    score_condition,
                    meal_category_for_scoring
                )
            candidate_copy['score'] = score_for_this_candidate
            
            if score_for_this_candidate is not None:
                candidates_with_ss_and_score.append(candidate_copy)
            else:
                print(f"Debug (Scen 3): Skipping {candidate_copy.get('FoodCode', 'N/A')} because score could not be calculated (was None).")

        # Normalize scores if there are any valid candidates
        if candidates_with_ss_and_score:
            scores = [c['score'] for c in candidates_with_ss_and_score]
            min_score = min(scores)
            max_score = max(scores)
            
            if max_score != min_score:
                for candidate in candidates_with_ss_and_score:
                    normalized_score = 100 * (candidate['score'] - min_score) / (max_score - min_score)
                    candidate['score'] = normalized_score
            else: # All scores are the same, normalize to 100 if positive, 0 if zero/negative
                for candidate in candidates_with_ss_and_score:
                    candidate['score'] = 100.0 if min_score > 0 else (0.0 if min_score == 0 else 0.0) # Or another logic
            print(f"Debug (Scen 3): Scores normalized to 0-100 range. Example: Min={min_score:.2f}, Max={max_score:.2f}")

        print(f"Debug (Scen 3): {len(candidates_with_ss_and_score)} candidates remaining after SS calculation and raw scoring/filtering.")

        if not candidates_with_ss_and_score:
            print("Seq Build (Scen 3): No candidates could be sized and scored in this iteration. Stopping sequence.")
            break

        candidates_with_ss_and_score.sort(
            key=lambda x: x.get('score', -float('inf')),
            reverse=True
        )
        print(f"Debug (Scen 3): Candidates sorted. Top candidate score: {candidates_with_ss_and_score[0].get('score', 'N/A'):.2f}")

        added_food_this_iteration = False
        for best_candidate_to_add in candidates_with_ss_and_score:
            candidate_actual_calories = _get_food_actual_calories(best_candidate_to_add)

            if candidate_actual_calories is None or candidate_actual_calories <= 0:
                print(f"Debug (Scen 3): Skipping {best_candidate_to_add.get('FoodCode', 'N/A')} due to invalid actual calories after SS calculation.")
                continue

            projected_total_calories = current_total_calories_in_sequence + candidate_actual_calories
            
            calorie_check_target = 0.0 # Initialize to ensure it always has a value
            
            if len(output_sequence) == 1: # This is for the second item
                calorie_check_target = target_calories_for_first_pair * 1.10 # Target for the pair (first two items)
                print(f"Debug (Scen 3): Checking 2nd item: {best_candidate_to_add.get('FoodCode', 'N/A')} (Score: {best_candidate_to_add.get('score', 'N/A'):.2f}) | Projected Total Cal={projected_total_calories:.1f} vs TargetForPair={calorie_check_target:.1f} (Current Seq Cal: {current_total_calories_in_sequence:.1f})")
                
                if projected_total_calories <= calorie_check_target:
                    output_sequence.append(best_candidate_to_add)
                    current_total_calories_in_sequence = projected_total_calories
                    current_food_node_for_rels = best_candidate_to_add
                    added_food_this_iteration = True
                    print(f"Debug (Scen 3): ADDED {best_candidate_to_add.get('FoodCode', 'N/A')} to sequence. New total cal: {current_total_calories_in_sequence:.1f}")
                    break # Break from candidate loop, move to next sequence iteration
                else:
                    # Attempt to reduce serving size to 0.5 if it's currently > 0.5
                    if best_candidate_to_add.get('serving_size') > 0.5:
                        print(f"Debug (Scen 3): {best_candidate_to_add.get('FoodCode', 'N/A')} EXCEEDS TargetForPair with current SS ({best_candidate_to_add.get('serving_size')}). Trying SS=0.5...")
                        temp_candidate_half_ss = best_candidate_to_add.copy()
                        temp_candidate_half_ss['serving_size'] = 0.5
                        candidate_actual_calories_half_ss = _get_food_actual_calories(temp_candidate_half_ss)
                        
                        if candidate_actual_calories_half_ss is None or candidate_actual_calories_half_ss <= 0:
                            print(f"Debug (Scen 3): Warn: SS=0.5 for {best_candidate_to_add.get('FoodCode', 'N/A')} resulted in invalid calories. Skipping this size.")
                            continue # Try next candidate
                        
                        projected_total_calories_half_ss = current_total_calories_in_sequence + candidate_actual_calories_half_ss
                        
                        if projected_total_calories_half_ss <= calorie_check_target:
                            output_sequence.append(temp_candidate_half_ss)
                            current_total_calories_in_sequence = projected_total_calories_half_ss
                            current_food_node_for_rels = temp_candidate_half_ss
                            added_food_this_iteration = True
                            print(f"Debug (Scen 3): ADDED {best_candidate_to_add.get('FoodCode', 'N/A')} with SS=0.5 to sequence. New total cal: {current_total_calories_in_sequence:.1f}")
                            break # Break from candidate loop, move to next sequence iteration
                        else:
                            print(f"Debug (Scen 3): {best_candidate_to_add.get('FoodCode', 'N/A')} (SS=0.5, Cal={candidate_actual_calories_half_ss:.1f}) still EXCEEDS TargetForPair ({projected_total_calories_half_ss:.1f} > {calorie_check_target:.1f}). Skipping this size.")
                            continue # Try next candidate
                    else: # Serving size is already 0.5 or less, and it still exceeds target
                        print(f"Debug (Scen 3): {best_candidate_to_add.get('FoodCode', 'N/A')} (SS={best_candidate_to_add.get('serving_size')}, Cal={candidate_actual_calories:.1f}) EXCEEDS TargetForPair and cannot reduce SS further. Skipping. (Projected {projected_total_calories:.1f} > {calorie_check_target:.1f})")
                        continue # Try next candidate
            
            else: # This is for 3rd and 4th items (or any item beyond the first pair)
                calorie_check_target = overall_meal_target_with_tolerance # Target for the entire meal
                print(f"Debug (Scen 3): Checking subsequent item: {best_candidate_to_add.get('FoodCode', 'N/A')} (Score: {best_candidate_to_add.get('score', 'N/A'):.2f}) | Projected Total Cal={projected_total_calories:.1f} vs OverallTarget={calorie_check_target:.1f} (Current Seq Cal: {current_total_calories_in_sequence:.1f})")
                
                if projected_total_calories <= calorie_check_target:
                    output_sequence.append(best_candidate_to_add)
                    current_total_calories_in_sequence = projected_total_calories
                    current_food_node_for_rels = best_candidate_to_add
                    added_food_this_iteration = True
                    print(f"Debug (Scen 3): ADDED {best_candidate_to_add.get('FoodCode', 'N/A')} to sequence. New total cal: {current_total_calories_in_sequence:.1f}")
                    break # Break from candidate loop, move to next sequence iteration
                else:
                    # Attempt to reduce serving size to 0.5 if it's currently > 0.5 for subsequent items too
                    if best_candidate_to_add.get('serving_size') > 0.5:
                        print(f"Debug (Scen 3): {best_candidate_to_add.get('FoodCode', 'N/A')} EXCEEDS OverallTarget with current SS ({best_candidate_to_add.get('serving_size')}). Trying SS=0.5...")
                        temp_candidate_half_ss = best_candidate_to_add.copy()
                        temp_candidate_half_ss['serving_size'] = 0.5
                        candidate_actual_calories_half_ss = _get_food_actual_calories(temp_candidate_half_ss)
                        
                        if candidate_actual_calories_half_ss is None or candidate_actual_calories_half_ss <= 0:
                            print(f"Debug (Scen 3): Warn: SS=0.5 for {best_candidate_to_add.get('FoodCode', 'N/A')} resulted in invalid calories. Skipping this size.")
                            continue # Try next candidate
                        
                        projected_total_calories_half_ss = current_total_calories_in_sequence + candidate_actual_calories_half_ss
                        
                        if projected_total_calories_half_ss <= calorie_check_target:
                            output_sequence.append(temp_candidate_half_ss)
                            current_total_calories_in_sequence = projected_total_calories_half_ss
                            current_food_node_for_rels = temp_candidate_half_ss
                            added_food_this_iteration = True
                            print(f"Debug (Scen 3): ADDED {best_candidate_to_add.get('FoodCode', 'N/A')} with SS=0.5 to sequence. New total cal: {current_total_calories_in_sequence:.1f}")
                            break # Break from candidate loop, move to next sequence iteration
                        else:
                            print(f"Debug (Scen 3): {best_candidate_to_add.get('FoodCode', 'N/A')} (SS=0.5, Cal={candidate_actual_calories_half_ss:.1f}) still EXCEEDS OverallTarget ({projected_total_calories_half_ss:.1f} > {calorie_check_target:.1f}). Skipping this size.")
                            continue # Try next candidate
                    else: # Serving size is already 0.5 or less, and it still exceeds target
                        print(f"Debug (Scen 3): {best_candidate_to_add.get('FoodCode', 'N/A')} (SS={best_candidate_to_add.get('serving_size')}, Cal={candidate_actual_calories:.1f}) EXCEEDS OverallTarget and cannot reduce SS further. Skipping. (Projected {projected_total_calories:.1f} > {calorie_check_target:.1f})")
                        continue # Try next candidate

        if not added_food_this_iteration:
            print("Seq Build (Scen 3): No candidate found in this iteration that meets calorie constraints. Stopping sequence.")
            break

    print(f"\n--- Scenario 3 Sequence Building Finished ---")
    print(f"Final Sequence Length: {len(output_sequence)} items")
    print(f"Final Total Calories in Sequence: {current_total_calories_in_sequence:.1f} (Meal Target was: {target_calories_for_meal:.1f})")
    
    return output_sequence
# Main function to call based on flag
#old version till 6/9
def get_food_recommendations(driver, flag, calorie_requirement, score_condition, meal_type=None,
                            input_food_dicts=None, required_conditions=None, required_allergens=None,
                            excluded_conditions=None, excluded_allergens=None, other_filters=None, eaten_list=None):
    print(f"\nReceived request with flag={flag}")

    # Ensure input_food_dicts is a list, even if None is passed
    if input_food_dicts is None:
        input_food_dicts = []
    
    # Convert FoodCode in eaten_list to string, as Neo4j might store FoodCode as string or int,
    # and comparison needs consistency. Assuming FoodCode in DB can be int or string.
    # If FoodCode is ALWAYS integer in DB, convert to int. If string, convert to string.
    # Based on "NOT toString({target_node_alias}.FoodCode) IN $excludedEatenFoodCodes AND NOT toInteger({target_node_alias}.FoodCode) IN $excludedEatenFoodCodes"
    # it seems there's uncertainty. Let's assume they are stored as integers primarily.
    processed_eaten_list = []
    if eaten_list:
        for code in eaten_list:
            try:
                processed_eaten_list.append(int(code)) # Assuming FoodCodes are primarily integers
            except (ValueError, TypeError):
                try:
                    processed_eaten_list.append(str(code)) # Fallback if int conversion fails
                except:
                     print(f"Warning: Could not process FoodCode '{code}' in eaten_list. Skipping this item.")
        print(f"Processed eaten_list: {processed_eaten_list}")


    # Convert FoodCode in input_food_dicts (if any)
    # For consistency, let's assume FoodCode should be an integer if possible
    for food_dict in input_food_dicts:
        if 'FoodCode' in food_dict and food_dict['FoodCode'] is not None:
            try:
                food_dict['FoodCode'] = int(food_dict['FoodCode'])
            except (ValueError, TypeError):
                 try:
                    food_dict['FoodCode'] = str(food_dict['FoodCode']) # Store as string if int fails
                    # print(f"Warning: FoodCode '{food_dict.get('FoodCode')}' in input_food_dicts converted to string as int failed.")
                 except:
                    print(f"Warning: Invalid FoodCode '{food_dict.get('FoodCode')}' in input_food_dicts. Could not convert to int or string. Setting to None.")
                    food_dict['FoodCode'] = None # Mark as invalid if all conversions fail


    if flag == 0:
        # This scenario was not requested to be fully defined here, but its call signature exists.
        print("Executing Scenario 1 (Standard Recommendations)...")
        return _execute_scenario_1(driver, calorie_requirement, score_condition, meal_type,
                                  input_food_dicts, required_conditions, required_allergens,
                                  excluded_conditions, excluded_allergens, other_filters)
        # print("Scenario 1 execution logic is not fully pasted here.")
        return None # Placeholder
    elif flag == 1:
        # This scenario was not requested to be fully defined here.
        print("Executing Scenario 2 (Sequence Building with one input)...")
        return _execute_scenario_2(driver, calorie_requirement, score_condition, meal_type,
                                  input_food_dicts, required_conditions, required_allergens,
                                  excluded_conditions, excluded_allergens, other_filters)
        # print("Scenario 2 execution logic is not fully pasted here.")
        return None # Placeholder
    elif flag == 2:
        print("Executing Scenario 3 (Sequence Building with Random First Food)...")
        return _execute_scenario_3(driver, calorie_requirement, score_condition, meal_type,
                                  None, # input_food_dicts is explicitly None for Scen 3 start
                                  required_conditions, required_allergens,
                                  excluded_conditions, excluded_allergens, other_filters,
                                  processed_eaten_list) # Pass the processed list
    else:
        print(f"Error: Invalid flag value '{flag}'. Must be 0, 1, or 2.")
        return None


if __name__ == '__main__':
    # --- Database Connection ---
    try:
        uri = NEO4J_URI
        user = NEO4J_USER
        password = NEO4J_PASSWORD
        driver = GraphDatabase.driver(uri, auth=(user, password))
        driver.verify_connectivity()
        print("Successfully connected to Neo4j.")
    except Exception as e:
        print(f"Failed to connect to Neo4j: {e}")
        driver = None
        exit()

    # --- Test Scenario 3 ---
    print("\n\n=== Testing Scenario 3 ===")
    if driver:
        # Example parameters for Scenario 3
        test_calorie_req_s3 = 1800
        test_score_cond_s3 = "Diabetes" # or "General" or "Hypertension" etc.
        test_meal_type_s3 = "DINNER"    # LUNCH, DINNER, BREAKFAST
        test_req_cond_s3 = [] # Example: ["Low FODMAP"]
        test_req_aller_s3 = []
        test_excl_cond_s3 = []
        test_excl_aller_s3 = [] # Example: ["Gluten"]
        test_other_filters_s3 = {"MealType": "Non-Vegetarian"} # Example: {"Cuisine": "Indian"}
        test_eaten_list_s3 = [9880022, 123456] # Example list of FoodCodes (as int or string) already eaten

        results_s3 = get_food_recommendations(
            driver,
            flag=2,
            calorie_requirement=test_calorie_req_s3,
            score_condition=test_score_cond_s3,
            meal_type=test_meal_type_s3,
            input_food_dicts=None, # Explicitly None for Scen 3
            required_conditions=test_req_cond_s3,
            required_allergens=test_req_aller_s3,
            excluded_conditions=test_excl_cond_s3,
            excluded_allergens=test_excl_aller_s3,
            other_filters=test_other_filters_s3,
            eaten_list=test_eaten_list_s3
        )

        if results_s3:
            print(f"\nScenario 3 Recommendations ({len(results_s3)} items):")
            final_total_cal = 0
            for i, food in enumerate(results_s3):
                food_name = food.get('foodName', food.get('FoodName', 'N/A'))
                food_code = food.get('FoodCode', 'N/A')
                serving_size = food.get('serving_size', 'N/A')
                actual_cals = _get_food_actual_calories(food)
                actual_cals_display = f"{actual_cals:.1f}" if actual_cals is not None else "N/A"
                if actual_cals is not None: final_total_cal += actual_cals
                score = food.get('score', 'N/S') # Not Scored
                score_display = f"{score:.2f}" if isinstance(score, float) else score

                print(f"  {i+1}. {food_name} (Code: {food_code}) - SS: {serving_size} - Est.Cal: {actual_cals_display} - Score: {score_display}")
                # print(f"     AssignedMC: {food.get('AssignedMC')}, AssignedMO: {food.get('AssignedMO')}") # For debugging MC/MO
            print(f"Total Estimated Calories for Scenario 3 Meal: {final_total_cal:.1f}")
        else:
            print("Scenario 3: No results or an error occurred.")

    # --- Close Driver ---
    if driver:
        driver.close()
        print("\nNeo4j connection closed.")