# -*- coding: utf-8 -*-

from typing import List, Dict, Any, Set
import json
from neo4j import GraphDatabase
from neo4j.exceptions import ServiceUnavailable

# ==============================================================================
# SECTION 1: CONFIGURATION
# ==============================================================================

# --- Neo4j Database Connection Details ---
# IMPORTANT: Replace these values with your actual database credentials.
# NEO4J_URI = "bolt://172.164.240.161:7687"
NEO4J_USER = "neo4j"
NEO4J_PASSWORD = "@MahilaMoghadami@"

# --- Test Name and Rule Configurations ---
TEST_NAME_MAPPING = {
    # ... (The rest of the mapping remains the same as before) ...
    "Hemoglobin A1c": ["Hemoglobin A1c", "Glycated Hemoglobin", "HBA1C", "HbA1c"],
    "Fasting Insulin": ["Insulin (fasting)", "Fasting Serum Insulin"],
    "Fasting Blood Glucose": ["Fasting Glucose", "FBG", "Fasting Blood Sugar"],
    "Random Blood Glucose": ["Random Glucose", "Non-fasting Glucose", "Random Blood Sugar"],
    "Glucose tolerance test": ["Oral Glucose Tolerance Test", "OGTT"],
    "Total Cholesterol": ["Total Cholesterol", "Chol", "Serum Cholesterol", "t.cholesterol", "Cholesterol"],
    "Triglycerides": ["Triglycerides", "TG", "Serum Triglycerides"],
    "HDL": ["HDL Cholesterol", "High-Density Lipoprotein", "HDL-C"],
    "LDL": ["LDL Cholesterol", "Low-Density Lipoprotein", "LDL-C", "Low Density Lipoprotein"],
    "LDL/HDL": ["LDL to HDL Ratio", "Cholesterol Ratio", "Atherogenic Index"],
    "Hemoglobin": ["Hb", "Hemoglobin Concentration", "HGB"],
    "RBC Indices": ["MCV", "MCH", "MCHC", "RDW", "Red Cell Indices", "RBC Indices"],
    "Iron": ["Serum Iron", "Fe", "Iron Level"],
    "Ferritin": ["Ferritin Level", "Serum Ferritin", "Iron Storage Protein"],
    "TIBC": ["Total Iron Binding Capacity", "TIBC", "Iron Binding Capacity"],
    "Transferrin Saturation Index": ["Transferrin Saturation", "%Transferrin Saturation", "TSAT"],
    "Creatinine": ["Creatinine", "Serum Creatinine", "Creatinine (Cr)", "Cr"],
    "Blood Urea Nitrogen": ["BUN", "Urea", "Blood Urea"],
    "eGFR": ["Estimated Glomerular Filtration Rate", "GFR", "eGFR (CKD-EPI)"],
    "Potassium": ["Potassium", "K+", "Hypokalemia Indicator"],
    "Total protein": ["Serum Total Protein", "Total Protein", "TP"],
    "Albumin": ["Albumin", "Alb", "Albumin Level"],
    "ALT/SGPT": ["Alanine Aminotransferase", "ALT", "SGPT"],
    "AST/SGOT": ["Aspartate Aminotransferase", "AST", "SGOT"],
    "TSH": ["Thyroid Stimulating Hormone", "TSH", "Thyrotropin"],
    "Anti-TPO": ["Anti-Thyroid Peroxidase", "Anti-TPO Antibody", "Anti-TPO"],
    "Vit D Total": ["Total Vitamin D", "25(OH)D", "Calcidiol", "Vitamin D", "25OH-Vitamin -D3"],
    "Vit B12": ["Vitamin B12", "Cobalamin", "B12"],
    "Calcium": ["Calcium", "Ca", "Serum Calcium"],
    "Magnesium": ["Magnesium", "Mg", "Serum Magnesium"],
    "IgE": ["Immunoglobulin E", "IgE", "Ig E"],
    "IgA": ["Immunoglobulin A", "IgA", "Ig A"],
    "Uric Acid": ["Uric Acid", "Serum Uric Acid", "UA", "Uricemia"],
}

DISEASE_INPUT_MAPPING = {
    "فشار خون بالا": "Hypertension",
    "بیماری خودایمنی": "MS",
    "IBS (سندروم روده تحریک پذیر)": "IBS",
    "IBS": "IBS",
    "بیماری قلبی": "CVD",
    "کبد چرب": "Nafld",
    "چربی خون بالا": "HighCholesterol",
    "دیابت": "Diabetes",
    "PCOS (سندروم تخمدان پلی‌کیستیک)": "PCOS",
    "PCOS": "PCOS",
    "نقرس": "Gout"
}

DISEASE_ACTIVATION_RULES = [
    # ... (Rules remain the same as before) ...
    {'test_name': 'Hemoglobin A1c', 'status': 'high', 'disease_tag': 'Diabetes'},
    {'test_name': 'Fasting Insulin', 'status': 'high', 'disease_tag': 'Diabetes'},
    {'test_name': 'Fasting Blood Glucose', 'status': 'high', 'disease_tag': 'Diabetes'},
    {'test_name': 'Random Blood Glucose', 'status': 'high', 'disease_tag': 'Diabetes'},
    {'test_name': 'Glucose tolerance test', 'status': 'high', 'disease_tag': 'Diabetes'},
    {'test_name': 'Total Cholesterol', 'status': 'high', 'disease_tag': 'HighCholesterol'},
    {'test_name': 'Triglycerides', 'status': 'high', 'disease_tag': 'hypoTriglycerides (TG)'},
    {'test_name': 'HDL', 'status': 'low', 'disease_tag': 'HDL'},
    {'test_name': 'LDL', 'status': 'high', 'disease_tag': 'HighCholesterol'},
    {'test_name': 'LDL/HDL', 'status': 'high', 'disease_tag': 'HighCholesterol'},
    {'test_name': 'Hemoglobin', 'status': 'low', 'disease_tag': 'Anemia'},
    {'test_name': 'RBC Indices', 'status': 'low', 'disease_tag': 'Anemia'},
    {'test_name': 'Iron', 'status': 'low', 'disease_tag': 'Anemia'},
    {'test_name': 'Ferritin', 'status': 'low', 'disease_tag': 'Anemia'},
    {'test_name': 'TIBC', 'status': 'low', 'disease_tag': 'Anemia'},
    {'test_name': 'Transferrin Saturation Index', 'status': 'low', 'disease_tag': 'Anemia'},
    {'test_name': 'Creatinine', 'status': 'high', 'disease_tag': 'Kidney'},
    {'test_name': 'Blood Urea Nitrogen', 'status': 'high', 'disease_tag': 'Kidney'},
    {'test_name': 'eGFR', 'status': 'low', 'disease_tag': 'Kidney'},
    {'test_name': 'ALT/SGPT', 'status': 'high', 'disease_tag': 'NAFLD'},
    {'test_name': 'AST/SGOT', 'status': 'high', 'disease_tag': 'NAFLD'},
    {'test_name': 'TSH', 'status': 'high', 'disease_tag': 'Hypothyroidism'},
    {'test_name': 'Anti-TPO', 'status': 'high', 'disease_tag': 'Hypothyroidism'},
    {'test_name': 'IgA', 'status': 'high', 'disease_tag': 'Celiac'},
    {'test_name': 'Uric Acid', 'status': 'high', 'disease_tag': 'Gout'},
]

COLOR_MAP = {0: 'Red', 1: 'Yellow', 2: 'Green'}


# ==============================================================================
# SECTION 2: HELPER & CORE LOGIC FUNCTIONS
# ==============================================================================

# --- The _create_reverse_mapping, _normalize_lab_results, _parse_disease_string,
# --- and _activate_disease_tags functions remain exactly the same as before.
# --- (They are omitted here for brevity but should be in your final .py file)

def _create_reverse_mapping(mapping: Dict[str, List[str]]) -> Dict[str, str]:
    """Creates a reverse mapping for fast test name normalization."""
    reverse_map = {}
    for standard_name, variations in mapping.items():
        for variation in variations:
            reverse_map[variation.lower()] = standard_name
    return reverse_map

_REVERSE_TEST_NAME_MAP = _create_reverse_mapping(TEST_NAME_MAPPING)

# ==============================================================================
# SECTION 2: HELPER & CORE LOGIC FUNCTIONS (Updated Function)
# ==============================================================================

def _parse_and_translate_diseases(diseases_string: str) -> List[str]:
    """
    Parses the '&&' separated disease string and translates each disease
    from the user-facing name to the internal system tag using DISEASE_INPUT_MAPPING.
    """
    if not diseases_string or not diseases_string.strip():
        return []
    
    # Split the input string into a list of Persian disease names
    persian_diseases = diseases_string.strip().split('&&')
    
    # Translate each Persian name to its corresponding standard tag
    standard_tags = []
    for disease_name in persian_diseases:
        # Use .strip() to remove any extra whitespace
        tag = DISEASE_INPUT_MAPPING.get(disease_name.strip())
        if tag:
            standard_tags.append(tag)
        else:
            # Optionally, log the name that couldn't be mapped
            print(f"Info: Input '{disease_name}' has no defined mapping and will be ignored.")

    return standard_tags

def _normalize_lab_results(lab_results: List[Dict[str, str]]) -> List[Dict[str, str]]:
    """Normalizes test names in the lab results list to a standard name."""
    normalized_results = []
    for test in lab_results:
        test_name_lower = test.get("test_name", "").lower()
        standard_name = _REVERSE_TEST_NAME_MAP.get(test_name_lower, test.get("test_name"))
        normalized_results.append({
            "test_name": standard_name,
            "status": test.get("status", "").lower()
        })
    return normalized_results


# def _parse_disease_string(diseases_string: str) -> List[str]:
#     """Parses the '&&' separated disease string into a list of diseases."""
#     if not diseases_string or not diseases_string.strip():
#         return []
#     return diseases_string.strip().split('&&')


def _activate_disease_tags(
    normalized_lab_results: List[Dict[str, str]],
    initial_diseases: List[str]
) -> Dict[str, List[str]]: # The return type is now a Dictionary
    """
    Determines the final list of active diseases and provides the reasons for their activation.
    Returns a dictionary where keys are disease tags and values are a list of activation reasons.
    """
    activation_details: Dict[str, List[str]] = {}

    # First, add diseases declared by the user
    for disease in initial_diseases:
        if disease not in activation_details:
            activation_details[disease] = []
        activation_details[disease].append("Declared by user")

    # Create a quick-lookup set for abnormal test results
    abnormal_tests = {
        (test['test_name'], test['status'])
        for test in normalized_lab_results
        if test['status'] not in ['normal', 'borderline', 'undefined']
    }

    # Now, check rules against lab results
    for rule in DISEASE_ACTIVATION_RULES:
        if (rule['test_name'], rule['status']) in abnormal_tests:
            disease_tag = rule['disease_tag']
            
            # Formulate the reason string for debugging
            reason = f"Triggered by lab result: '{rule['test_name']}' was '{rule['status']}'"
            
            # Add the disease and its activation reason to the dictionary
            if disease_tag not in activation_details:
                activation_details[disease_tag] = []
            
            # Avoid adding duplicate reasons if a rule somehow appears twice
            if reason not in activation_details[disease_tag]:
                 activation_details[disease_tag].append(reason)
                 
    return activation_details
class Neo4jConnection:
    """Manages the connection to the Neo4j database."""
    def __init__(self, uri, user, password):
        self.driver = GraphDatabase.driver(uri, auth=(user, password))

    def close(self):
        self.driver.close()

    def fetch_all_foods(self) -> List[Dict[str, Any]]:
        """
        Fetches all 'Food' nodes and returns them as a list of dictionaries.
        """
        query = "MATCH (f:Food) RETURN f"
        try:
            with self.driver.session() as session:
                result = session.run(query)
                # Convert each Neo4j Node object in the result to a standard dictionary
                return [dict(record["f"]) for record in result]
        except ServiceUnavailable as e:
            # print(f"Error: Unable to connect to Neo4j at {NEO4J_URI}. Exception: {e}")
            print(f"Error: Unable to connect to Neo4j. Exception: {e}")
            # Return an empty list or raise the exception, depending on desired behavior
            return []
        except Exception as e:
            print(f"An unexpected error occurred during database query: {e}")
            return []

def _calculate_food_color(food_node: Dict[str, Any], active_diseases: List[str]) -> str:
    """
    Calculates the final color for a single food item based on its intrinsic
    'Healthy_level' and then applying rules for active diseases.
    """
    # Step 1: Get the base health level from the food's 'Healthy_level' property.
    # Default to 2 (Green) if the property is missing or invalid, ensuring safety.
    healthy_level_raw = food_node.get('Healthy_level', 2)
    try:
        healthy_level = int(healthy_level_raw)
    except (ValueError, TypeError):
        healthy_level = 2  # Default to Green if the value is not a valid integer

    # Step 2: Apply the new logic rules.
    
    # Rule 1: If the food is intrinsically 'Red', it's always red.
    if healthy_level == 0:
        return COLOR_MAP[0]  # Returns 'Red'

    # For levels 1 and 2, the initial level is the healthy_level.
    # This correctly sets the ceiling (it can't get better than its base level).
    final_level = healthy_level

    # Step 3: Apply disease-specific rules, which can only downgrade the color.
    # This loop is the same as before.
    for disease in active_diseases:
        prop_name = f"Disease_{disease}_Level"
        disease_level_raw = food_node.get(prop_name, 2) # Default to 2 (no restriction)
        
        try:
            disease_level = int(disease_level_raw)
        except (ValueError, TypeError):
            disease_level = 2
            
        final_level = min(final_level, disease_level)

    return COLOR_MAP.get(final_level, 'Green')

# ==============================================================================
# SECTION 3: MAIN PIPELINE FUNCTION
# ==============================================================================
def process_recommendations(
    diseases_string: str,
    lab_results: List[Dict[str, str]]
) -> List[Dict[str, Any]]:
    """
    Main pipeline to generate food recommendations.
    """
    # Step 1: Pre-process inputs
    # initial_diseases = _parse_and_translate_diseases(diseases_string)
    # normalized_lab_results = _normalize_lab_results(lab_results)

    # # Step 2: Activate disease tags and get the detailed reasons
    # disease_activation_details = _activate_disease_tags(normalized_lab_results, initial_diseases)
    
    # # --- NEW DEBUGGING SECTION ---
    # print("\n" + "="*25)
    # print("Disease Activation Report:")
    # print(json.dumps(disease_activation_details, indent=2, ensure_ascii=False))
    # print("="*25 + "\n")
    # # --- END OF DEBUGGING SECTION ---

    # # Extract just the list of disease names for the next step
    # final_active_diseases = list(disease_activation_details.keys())
    # print(f"Final Active Diseases for processing: {final_active_diseases}")
    
    # # Step 3: Fetch all food data from the database
    # db_connection = Neo4jConnection(NEO4J_URI, NEO4J_USER, NEO4J_PASSWORD)
    # all_foods = db_connection.fetch_all_foods()
    # db_connection.close()
    
    # if not all_foods:
    #     print("Warning: No food data was fetched from the database. Returning empty list.")
    #     return []

    # # Step 4: Process each food to determine its color
    # recommendations = []
    # for food in all_foods:
    #     # The _calculate_food_color function doesn't need to change.
    #     # It correctly uses the list of active diseases.
    #     final_color = _calculate_food_color(food, final_active_diseases)
        
    #     recommendations.append({
    #         "FoodCode": food.get("FoodCode"),
    #         "Foodname": food.get("FoodNamePersian"),
    #         "color": final_color,
    #         "Calories": food.get("Calories"),
    #         "protein": food.get("Protein_g"),
    #         "carbohydrate": food.get("Carbohydrates_g"),
    #         "Fat": food.get("Fat_g"),
    #         "EnglishFoodName": food.get("EnglishFoodName")
    #     })
        
    # return recommendations
    
    # Neo4j connection disabled - return empty list
    return []
# ==============================================================================
# SECTION 4: EXAMPLE USAGE
# ==============================================================================

# if __name__ == '__main__':
#     sample_diseases_string = "PCOS (سندروم تخمدان پلی‌کیستیک)"
#     sample_lab_results = [
#         {"test_name": "MCHC", "status": "low"},
#         {"test_name": "Alanine Aminotransferase", "status": "high"},
#     ]

#     print("Starting pipeline with sample data...")
#     final_recommendations = process_recommendations(sample_diseases_string, sample_lab_results)

#     print("-" * 20)
#     print("Final Recommendations Output:")
#     print(json.dumps(final_recommendations, indent=4, ensure_ascii=False))