import jdatetime
from django.db import models
from datetime import datetime, date, timedelta
from django.contrib.auth.models import User as DjangoUser
from typing import List
from django.contrib.auth.models import UserManager
from django.contrib.auth.models import Group
from django.core.cache import cache
import os
from django.db.models import Avg, Max, Count, Q
from django.utils.timezone import now
from django.db.models.signals import post_save
from django.dispatch import receiver
from panel.tagging import process_questionnaire
from django.utils.text import slugify
from django.conf import settings


########### user models ############


class TempToken(models.Model):
    temp_token = models.TextField(null=True)
    phone_number = models.TextField(null=True)
    email = models.TextField(default=None, null=True)
    activation_code = models.TextField()
    time = models.DateTimeField(default=datetime.now)  # The moment the code is updated
    sms_sent_time = models.DateTimeField(default=datetime.now)  # The moment the message is sent
    email_sent_time = models.DateTimeField(default=datetime.now)  # The moment the message is sent


class Insulin(models.Model): 
    Data_Base_Number = models.TextField(null=False)
    name = models.TextField()
    Company = models.TextField()
    EffectType = models.IntegerField()  # Bolus or short = 1 , Basal or long = 2 , short/long = 3
    coefficient = models.IntegerField(default=0)

    def __str__(self):
        return self.name


class GLP1(models.Model): 
    name = models.TextField(default="-")
    EN_Name = models.TextField(default="-")
    description = models.TextField(default="")
    coefficient = models.TextField(default="")
    DataBaseNumber = models.IntegerField(null=False)

    def __str__(self):
        return self.name


class Address(models.Model):
    address = models.TextField(null=True, blank=True, default=None)
    city = models.TextField(null=True, blank=True, default=None)
    postal_code = models.TextField(null=True, blank=True, default=None)
    latitude = models.FloatField(null=True, blank=True, default=None)
    longitude = models.FloatField(null=True, blank=True, default=None)


class Profile(models.Model):
    first_name = models.TextField(default="")
    last_name = models.TextField(default="")
    gender = models.BooleanField(choices=((0, 'مرد'), (1, 'زن')), default=0)
    national_id = models.TextField(null=True, blank=True, default=None)
    birth_date = models.DateTimeField(default=datetime.now)
    weight = models.FloatField(default=1)
    goal_weight = models.IntegerField(default=1)
    height = models.IntegerField(default=1)
    diabetes_type = models.IntegerField(choices=((0, 'بارداری'),
                                                 (1, 'type1'),
                                                 (2, 'type2')), default=1)
    
    pregnancy_status = models.IntegerField(choices=((0, 'باردار نیست'), (1, 'باردار')), default=0)
    _diseases = models.TextField(default=None, null=True, blank=True)  # 1&2&5&5
    DISEASE_CHOICES = {  # the list of disease for each number
        0: 'دیابت/پیش دیابت',
        1: 'چاقی',
        2: 'کبد چرب',
        3: 'فشار خون',
        4: 'هایپرلیپیدمی/چربی خون',
        5: 'PCOS',
        6: 'IBS',
        7: 'میگرن',
        8: 'Healthy',
        9: 'IBD',
        10: 'Gout',
        11: 'Kidney Disease',
        12: 'Celiac',
        13: 'Anemia',
        14: 'Hypothyroidism',
        15: 'MS',
        # Add more disease codes and their corresponding strings here
    }
    detection_year = models.IntegerField(default=1398)
    minimum_sugar = models.IntegerField(default=80)
    maximum_sugar = models.IntegerField(default=180)
    hypo = models.IntegerField(default=70)
    hyper = models.IntegerField(default=240)
    user_setting = models.TextField(default="")
    weight_state = models.TextField(default="")
    insulin_short_effect = models.ForeignKey(Insulin, on_delete=models.SET_NULL, related_name='short_effect', null=True,
                                             blank=True)
    insulin_long_effect = models.ForeignKey(Insulin, on_delete=models.SET_NULL, related_name='long_effect', null=True,
                                            blank=True)
    insulin_mixed_effect = models.ForeignKey(Insulin, on_delete=models.SET_NULL, related_name='mixed_effect', null=True,
                                             blank=True)
    glp1 = models.ForeignKey(GLP1, on_delete=models.SET_NULL, null=True, blank=True)
    insulin_sensitivity = models.IntegerField(default=0)
    type_of_treatment = models.IntegerField(default=0)  # 0->drug  1->insulin 2->GLP1 and other combinations of them ...
    pills = models.TextField(default="&")
    activity_level = models.FloatField(
        choices=((1, 'خیلی کم'), (1.25, 'کم'), (1.5, 'متوسط'), (1.75, 'زیاد'), (2, 'خیلی زیاد')), default=1.5)
    CR = models.FloatField(default=0)
    CR_coach = models.FloatField(default=0)
    BMR = models.FloatField(default=0)
    Carbohydrates_g = models.FloatField(default=0)
    Protein_g = models.FloatField(default=0)
    Fat_g = models.FloatField(default=0)
    Fiber_g = models.FloatField(default=0)
    Carbohydrates_unit = models.FloatField(default=0)
    Protein_unit = models.FloatField(default=0)
    Fat_unit = models.FloatField(default=0)
    Fiber_unit = models.FloatField(default=0)
    Carbohydrates_distribution_list_g = models.TextField(default="&0&0&0&0&0&0&")
    Protein_distribution_g = models.FloatField(default=0)
    Fat_distribution_g = models.FloatField(default=0)
    Fiber_distribution_g = models.FloatField(default=0)
    FavoritFoods = models.TextField(default="&")
    FavoritActivities = models.TextField(default="&")
    FavoritDrugs = models.TextField(default="&")
    Carb_rate = models.FloatField(default=0)
    a1c = models.FloatField(default=7)
    image = models.ImageField(upload_to='panel/profiles/patients/', null=True, default=None, blank=True)
    device_type = models.TextField(null=True, default="", blank=True)
    version = models.TextField(null=True, blank=True)
    market = models.TextField(null=True, blank=True)
    email = models.EmailField(null=True, blank=True)
    user_code = models.TextField(null=True, blank=True)  # aaab9
    intro_code = models.TextField(null=True, blank=True)  # aaab9
    sub_users = models.TextField(null=True, default="&")  # &abcde&obcdh&lbcdk&
    register_time = models.DateTimeField(null=True, default=None, blank=True)
    avatar = models.IntegerField(default=0, blank=True)
    foods_version = models.DateTimeField(null=True, default=None, blank=True)
    firebase_token = models.TextField(null=True, default=None, blank=True)
    therapist = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, default=None, blank=True)
    address = models.ForeignKey(Address, on_delete=models.SET_NULL, null=True, default=None, blank=True)
    coins = models.IntegerField(default=0, blank=True, null=True)
    coins_history = models.JSONField(default=dict, blank=True, null=True)
    last_login = models.DateTimeField(null=True, blank=True)
    diet_tags = models.JSONField(default=dict, blank=True, null=True)
    hospital_code = models.TextField(null=True, blank=True, default=None)  # Hospital tracking field
    
    # Acquisition tracking fields
    ACQUISITION_CHOICES = [
        ('friends', 'دوستان'),
        ('doctor', 'پزشک'),
        ('instagram', 'اینستاگرام'),
        ('myket', 'مایکت'),
        ('bazar', 'بازار'),
        ('internet', 'اینترنت'),
        ('other', 'دیگر'),
    ]
    acquisition_source = models.CharField(max_length=50, null=True, blank=True, choices=ACQUISITION_CHOICES)
    acquisition_details = models.TextField(null=True, blank=True)  # Additional context about acquisition
    acquisition_date = models.DateTimeField(null=True, blank=True)  # When they first heard about us

    # coins_history = {
    #     "history": [
    #         {
    #             "current_coins": 12,
    #             "type": "app usage",
    #             "datetime": timestamp
    #         },
    #         {
    #             "current_coins": 112,
    #             "type": "",
    #             "datetime": timestamp
    #         }
    #     ]
    # }
    def add_coins(self, amount, _type='app'):
        self.coins += amount
        history = {
            'current_coins': self.coins,
            'type': _type,
            'datetime': datetime.now().timestamp()
        }
        if "history" not in self.coins_history:
            self.coins_history["history"] = []
        self.coins_history['history'].append(history)
        self.save()

    @property
    def age(self):
        years = int((datetime.now() - self.birth_date).days / 365)
        return years

    # for making int to string from list
    @property
    def diseases(self) -> List[str]:
        if self._diseases is None:
            return []
        ds_list = self._diseases.split('&')
        ds_list = [self.DISEASE_CHOICES.get(int(disease.strip()), 'Unknown') for disease in ds_list if disease.strip()]
        return ds_list

    def write_diseases(self, diseases_list: List[int]):
        if len(diseases_list) == 0:
            self._diseases = None
        else:
            self._diseases = '&'.join(set(diseases_list))  # unique diseases
        self.save()

    def add_to_diseases(self, new_disease: int):
        diseases_set = set(self.diseases)
        diseases_set.add(new_disease)
        self.write_diseases(list(diseases_set))

    def have_disease(self, disease: int):
        return disease in self.diseases


class User(models.Model):
    phone_number = models.TextField()
    token = models.TextField()
    profile = models.ForeignKey(Profile, on_delete=models.CASCADE, null=True, blank=True)
    created_at = models.DateField(default=date.today)
    isLogedin = models.BooleanField(default=True)
    isRegistered = models.BooleanField(default=False)
    is_favorite = models.BooleanField(default=False)
    django_user = models.ForeignKey(DjangoUser, on_delete=models.CASCADE, null=True)
    
    # B2B relationship - will be set programmatically based on registration patterns
    # Users are now linked to companies via ZiluckGroup through django_user.groups relationship

    def __str__(self):
        return self.identifier

    @property
    def identifier(self):
        if self.profile is not None:
            i = self.profile.first_name + " " + self.profile.last_name + " " + str(self.id)
        elif self.phone_number is not None:
            i = self.phone_number + " " + str(self.id)
        else:
            i = str(self.id)
        return i

    @property
    def churn_risk_score(self):
        """Calculate churn risk score based on user activity and engagement"""
        score = 0
        risk_level = "Low"
        
        try:
            # Check last login activity
            if self.profile and self.profile.last_login:
                days_since_login = (datetime.now() - self.profile.last_login).days
                if days_since_login > 30:
                    score += 3
                elif days_since_login > 14:
                    score += 2
                elif days_since_login > 7:
                    score += 1
            else:
                score += 4  # No login recorded
            
            # Check recent activity points
            try:
                recent_points = Points.objects.filter(user=self, time__gte=datetime.now() - timedelta(days=30)).first()
                if recent_points:
                    if recent_points.total_points < 10:
                        score += 2
                    elif recent_points.total_points < 50:
                        score += 1
                else:
                    score += 3  # No recent activity
            except:
                score += 2
            
            # Check subscription status
            try:
                from panel.models import TherapistSelection
                subscription = TherapistSelection.objects.filter(user=self.django_user).last()
                if subscription and not subscription.has_credit():
                    score += 2  # Expired subscription
            except:
                pass
            
            # Check data logging frequency
            recent_logs_count = 0
            try:
                # Count recent food, activity, sugar logs
                recent_logs_count += Food_eating_log.objects.filter(user=self, time__gte=datetime.now() - timedelta(days=14)).count()  # pyright: ignore[reportUndefinedVariable]  # noqa: F821
                recent_logs_count += Activities_log.objects.filter(user=self, start_date_time__gte=datetime.now() - timedelta(days=14)).count()
                recent_logs_count += SugarMeasurement.objects.filter(user=self, time__gte=datetime.now() - timedelta(days=14)).count()
                
                if recent_logs_count == 0:
                    score += 3
                elif recent_logs_count < 3:
                    score += 1
            except:
                score += 1
            
            # Determine risk level
            if score >= 8:
                risk_level = "High"
            elif score >= 5:
                risk_level = "Medium"
            else:
                risk_level = "Low"
                
        except Exception as e:
            print(f"Error calculating churn risk for user {self.id}: {e}")
            risk_level = "Unknown"
        
        return risk_level
    
    @property
    def lacto_points_score(self):
        """Get user's latest Lacto Points score"""
        try:
            latest_points = Points.objects.filter(user=self).order_by('-time').first()
            return latest_points.total_points if latest_points else 0
        except:
            return 0
    
    @property
    def latest_biometrics(self):
        """Get user's latest biometric readings"""
        biometrics = {}
        try:
            # Latest weight
            latest_weight = Weights.objects.filter(user=self).order_by('-time').first()
            if latest_weight:
                biometrics['weight'] = latest_weight.weight
                biometrics['bmi'] = latest_weight.BMI
            
            # Latest blood sugar
            latest_sugar = SugarMeasurement.objects.filter(user=self).order_by('-time').first()
            if latest_sugar:
                biometrics['blood_sugar'] = latest_sugar.suger_amount
            
            # Latest A1C
            latest_a1c = AOneC.objects.filter(user=self).order_by('-time').first()
            if latest_a1c:
                biometrics['a1c'] = latest_a1c.percentage
                
        except:
            pass
        
        return biometrics
    
    @property
    def subscription_info(self):
        """Get user's current subscription information"""
        subscription_data = {
            'status': 'Free',
            'plan': 0,
            'start_date': None,
            'end_date': None,
            'renewal_status': 'N/A'
        }
        
        try:
            from panel.models import TherapistSelection
            from payment.models import ZarinPalPayment
            
            # Get latest subscription
            subscription = TherapistSelection.objects.filter(user=self.django_user).last()
            if subscription:
                subscription_data['start_date'] = subscription.created_at
                subscription_data['end_date'] = subscription.credit
                subscription_data['status'] = 'Active' if subscription.has_credit() else 'Expired'
                subscription_data['renewal_status'] = 'Active' if subscription.has_credit() else 'Needs Renewal'
            
            # Get plan info
            latest_payment = ZarinPalPayment.objects.filter(user=self, is_payed=True).order_by('-created_at').first()
            if latest_payment:
                subscription_data['plan'] = latest_payment.plan
                
        except:
            pass
            
        return subscription_data

    def calculate_cr_macro_nutrients_distribution(self):
        BMR = 0
        BMI = 20
        CR = 0
        CR_coach = 0

        age = datetime.now().year - self.profile.birth_date.year
        weight = self.profile.weight
        height = self.profile.height
        gender = self.profile.gender
        # activity_level = self.profile.activity_level
        activity_level = 1.3
        type_of_treatment = self.profile.type_of_treatment
        metabolism_Energy = 1
        boy = [15, 15, 15,
               16, 16, 16,
               15, 15, 15, 15,
               14, 14, 14, 14,
               13.5, 13.5, 13.5, 13.5]
        girl = [15, 15, 15,
                16, 16, 16,
                17, 17, 17, 17,
                16, 16, 16, 16,
                17, 17, 17, 17]

        if age < 19 and (age > 1) and (gender == False):  # man
            CR = boy[age - 1] * height
            BMR = CR
            L = round(((height / 100) ** 2) * 18.5, 1)
            H = round(((height / 100) ** 2) * 25, 1)
            BMI = weight / ((height / 100) ** 2)
            if BMI > 25:
                IBW = H
            elif BMI < 18.5:
                IBW = L
            else:
                IBW = weight
            weight_state = "شما در سن رشد هستید و محدوده وزن حالت سلامت برای شما " + str(L) + " تا " + str(
                H) + " کیلوگرم میباشد"
        elif age < 19 and (age > 1) and (gender == True):  # woman
            CR = girl[age - 1] * height
            BMR = CR
            L = round(((height / 100) ** 2) * 18.5, 1)
            H = round(((height / 100) ** 2) * 25, 1)
            BMI = weight / ((height / 100) ** 2)
            if BMI > 25:
                IBW = H
            elif BMI < 18.5:
                IBW = L
            else:
                IBW = weight
            weight_state = "شما در سن رشد هستید و محدوده وزن حالت سلامت برای شما " + str(L) + " تا " + str(
                H) + " کیلوگرم میباشد"
        else:
            BMI = weight / ((height / 100) ** 2)
            if BMI < 18.5:
                CR = 500  # to adding weight
                IBW = ((height / 100) ** 2) * 22
                weight_state = "تلاش کنید به محدوده وزن ایده آل نزدیک شوید، وزن حالت سلامت برای شما " + str(
                    int(IBW)) + " کیلوگرم میباشد"

            elif BMI < 25:
                CR = 0  # Normal range
                IBW = weight
                weight_state = "وزن شما در محدوده نرمال قرار دارد"
            elif BMI <= 30:
                CR = -500  # to reduce weight
                if gender is False:  # men
                    IBW = ((height / 100) ** 2) * 23
                else:  # woman
                    IBW = ((height / 100) ** 2) * 22
                weight_state = "شما اضافه وزن دارید و تلاش کنید به وزن حالت سلامت برسید، وزن حالت سلامت برای شما " + str(
                    int(IBW)) + " کیلوگرم میباشد"
            else:  # BMI upper than 30
                CR = -500  # to reduce weight
                IBW = ((height / 100) ** 2) * 24
                weight_state = "شما چاق محسوب میشوید و برای کنترل بهتر است از وزن خود بکاهید و به وزن حالت سلامت خود یعنی " + str(
                    int(IBW)) + " کیلو گرم نزدیک شوید"

            if gender is False:  # men
                BMR = (10 * weight) + (6.25 * height) - (5 * age) + 5
            else:  # woman
                BMR = (10 * weight) + (6.25 * height) - (5 * age) - 161

            if BMI < 18.5:
                metabolism_Energy = 1.1
            CR = CR + round(BMR * activity_level * metabolism_Energy, 2)
        CR_coach = CR
        carbohydrates_g = round((CR * 0.53) / 4, 2)
        protein_g = round((CR * 0.2) / 4, 2)
        fat_g = round((CR * 0.27) / 9, 2)

        fiber_g = 0
        if gender is False:  # men
            fiber_g = 38
        else:  # women
            fiber_g = 25

        carbohydrates_coefficient = [[0.15, 0.13, 0.24, 0.13, 0.24, 0.11], [0.15, 0.13, 0.22, 0.13, 0.22, 0.15]]
        carbohydrate_distribution_g = []
        carbohydrate_distribution_unit = []
        if type_of_treatment == 0:
            set = carbohydrates_coefficient[0]  # for drug
        elif type_of_treatment == 1:
            set = carbohydrates_coefficient[1]  # for insulin
        else:
            set = carbohydrates_coefficient[0]  # for glp1 type and combination models ...

        for x in set:
            carbohydrate_distribution_g.append(round(x * carbohydrates_g, 2))
            carbohydrate_distribution_unit.append(round(x * carbohydrates_g / 15, 2))
        data = {
            "CR": CR,
            "CR_coach": CR_coach,
            "BMR": BMR,
            "IBW": int(IBW),
            "weight_state": weight_state,
            "macro_nutrients": {
                "carbohydrates_g": round(carbohydrates_g, 2),
                "carbohydrates_unit": round(carbohydrates_g / 15, 2),
                "protein_g": round(protein_g, 2),
                "protein_unit": round(protein_g / 7, 2),
                "fiber_g": round(fiber_g, 2),  # need_changes
                "fiber_unit": round(fiber_g / 7, 2),  # need_changes
                "fat_g": round(fat_g, 2),
                "fat_unit": round(fat_g / 6, 2)
            },
            "carbohydrate_distribution_g": carbohydrate_distribution_g,
            "carbohydrate_distribution_unit": carbohydrate_distribution_unit,
            "protein_distribution_g": round(protein_g / 6, 2),
            "protein_distribution_unit": round((protein_g / 6) / 7, 2),
            "fat_distribution_g": round((fat_g / 6), 2),
            "fat_distribution_unit": round(((fat_g / 6) / 6), 2),
            "fiber_distribution_g": round((fiber_g / 6), 2),  # need_changes
            "fiber_distribution_unit": round(((fiber_g / 6) / 6), 2)  # need_changes
        }
        return data


class Report(models.Model):
    code = models.TextField(null=True)  # unique code for report
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    created_at = models.DateTimeField(default=datetime.now)
    start_time = models.DateField(null=True)
    end_time = models.DateField(null=True)
    message = models.TextField(null=True)


class Activity(models.Model):
    name = models.TextField()
    Metz = models.FloatField(default=0)

    def __str__(self):
        return self.name


class Food(models.Model):  # new
    Data_Base_Number = models.TextField(default="")
    FA_Name = models.TextField(default="نامعلوم")
    Protein_g = models.FloatField(default=0, db_index=True)
    Fat_g = models.FloatField(default=0, db_index=True)
    Carbohydrates_g = models.FloatField(default=0, db_index=True)   
    AL_List = models.TextField(default="") # Allergens List
    CS_List = models.TextField(default="") # Controlled Substances List
    HEI_1_Fruit = models.FloatField(default=0)
    HEI_2_Fruit = models.FloatField(default=0)
    HEI_1_Vegetables = models.FloatField(default=0)
    HEI_2_Vegetables = models.FloatField(default=0)
    HEI_Whole_Grain_Grains = models.FloatField(default=0)
    HEI_Dairy = models.FloatField(default=0)
    HEI_Protein_Foods = models.FloatField(default=0)
    HEI_Seafood_and_Plant_Proteins = models.FloatField(default=0)
    HEI_Refined_Grains = models.FloatField(default=0)
    HEI_Sodium = models.FloatField(default=0)
    HEI_Sugar = models.FloatField(default=0)
    Saturated_Fat_g = models.FloatField(default=0)
    dairy_unit = models.FloatField(default=0)
    fat_unit = models.FloatField(default=0)
    For_calculation = models.FloatField(default=0)
    cereal_unit = models.FloatField(default=0)
    meat_unit = models.FloatField(default=0)
    fruit_unit = models.FloatField(default=0)
    vegetables_unit = models.FloatField(default=0)
    Ash_g = models.FloatField(default=0)
    Calories = models.FloatField(default=0, db_index=True)
    Starch_g = models.FloatField(default=0)
    Sucrose_g = models.FloatField(default=0)
    Glucose_g = models.FloatField(default=0)
    Fructose_g = models.FloatField(default=0)
    Lactose_g = models.FloatField(default=0)
    Maltose_g = models.FloatField(default=0)
    Alcohol_g = models.FloatField(default=0)
    Water_g = models.FloatField(default=0)
    Caffeine_mg = models.FloatField(default=0)
    Theobromine_mg = models.FloatField(default=0)
    Sugar_g = models.FloatField(default=0)
    Galactose_g = models.FloatField(default=0)
    Fiber_g = models.FloatField(default=0)
    Calcium_mg = models.FloatField(default=0)
    Iron_mg = models.FloatField(default=0)
    Magnesium_mg = models.FloatField(default=0)
    Phosphorus_mg = models.FloatField(default=0)
    Potasssium_mg = models.FloatField(default=0)
    Sodium_mg = models.FloatField(default=0)
    Zinc_mg = models.FloatField(default=0)
    Cupper_mg = models.FloatField(default=0)
    Fluoride_mcg = models.FloatField(default=0)
    Manganese_mg = models.FloatField(default=0)
    Selenium_mcg = models.FloatField(default=0)
    Vitamin_A_IU = models.FloatField(default=0)
    Retinol_mcg = models.FloatField(default=0)
    Retinol_Equivalents_mcg = models.FloatField(default=0)
    Beta_Carotene_mcg = models.FloatField(default=0)
    Alpha_Carotene_mcg = models.FloatField(default=0)
    Vitamin_E_mg = models.FloatField(default=0)
    Vitamin_D_mcg = models.FloatField(default=0)
    Vitamin_D2_Ergocalciferol_mcg = models.FloatField(default=0)
    Vitamin_D3_Cholecalciferol_mcg = models.FloatField(default=0)
    Beta_Cryptoxanthin_mcg = models.FloatField(default=0)
    Lycopene_mcg = models.FloatField(default=0)
    Lutein_and_Zeaxanthin_mcg = models.FloatField(default=0)
    Vitamin_C_mg = models.FloatField(default=0)
    Vitamin_B1_mg = models.FloatField(default=0)
    Vitamin_B2_mg = models.FloatField(default=0)
    Vitamin_B3_mg = models.FloatField(default=0)
    Vitamin_B5_mg = models.FloatField(default=0)
    Vitamin_B6_mg = models.FloatField(default=0)
    Vitamin_B9_mg = models.FloatField(default=0)
    Vitamin_B12 = models.FloatField(default=0)
    Choline_mg = models.FloatField(default=0)
    Cholesterol_mg = models.FloatField(default=0)
    MUFA = models.FloatField(default=0)
    PUFA = models.FloatField(default=0)
    Name_of_home_units = models.TextField(default="&گرم&")
    Weight_of_home_units = models.TextField(default="&1&")
    Index_of_home_units = models.TextField(default="&0&")
    english_Name = models.TextField(default="")
    arabic_Name = models.TextField(default="")
    microbiome_score = models.FloatField(null=True, default=0)
    priority = models.IntegerField(default=10)

    # New Boolean Fields
    Dessert = models.BooleanField(default=False)
    Added_Meat = models.BooleanField(default=False)
    Ing_Main = models.BooleanField(default=False)
    Beverage = models.BooleanField(default=False)
    Sandwich = models.BooleanField(default=False)
    Condiment = models.BooleanField(default=False)
    Lunch = models.BooleanField(default=False)
    Snack = models.BooleanField(default=False)
    Bread = models.BooleanField(default=False)
    Entree = models.BooleanField(default=False)
    Soup = models.BooleanField(default=False)
    Side_Dish = models.BooleanField(default=False)
    Appetizer = models.BooleanField(default=False)
    Dinner = models.BooleanField(default=False)
    Pasta = models.BooleanField(default=False)
    Breakfast = models.BooleanField(default=False)
    Salad = models.BooleanField(default=False)
    Ingredient = models.BooleanField(default=False)
    Bacteria_Acidaminococcus = models.BooleanField(default=False)
    Bacteria_Bacteroidetes = models.BooleanField(default=False)
    Bacteria_Alistipes = models.BooleanField(default=False)
    Bacteria_Bifidobacterium = models.BooleanField(default=False)
    Bacteria_Akkermansia = models.BooleanField(default=False)
    Bacteria_Blautia = models.BooleanField(default=False)
    Bacteria_Fecalibacterium = models.BooleanField(default=False)
    Bacteria_Lawsonibacter_asaccharolyticus = models.BooleanField(default=False)
    Bacteria_Lactobacillus = models.BooleanField(default=False)
    Vegan = models.BooleanField(default=False)
    Umami = models.BooleanField(default=False)
    Slow_cooked = models.BooleanField(default=False)
    Spicy = models.BooleanField(default=False)
    Baked = models.BooleanField(default=False)
    Pickled = models.BooleanField(default=False)
    Low_calorie = models.BooleanField(default=False)
    Grilled = models.BooleanField(default=False)
    Roasted = models.BooleanField(default=False)
    High_protein = models.BooleanField(default=False)
    Fried = models.BooleanField(default=False)
    Sweet = models.BooleanField(default=False)
    Salty = models.BooleanField(default=False)
    Lactose_free = models.BooleanField(default=False)
    Low_carb = models.BooleanField(default=False)
    Cold = models.BooleanField(default=False)
    Vegetarian = models.BooleanField(default=False)
    Sour = models.BooleanField(default=False)
    Sweet_Sour = models.BooleanField(default=False)
    Raw = models.BooleanField(default=False)
    Gluten_free = models.BooleanField(default=False)
    Boiled = models.BooleanField(default=False)
    Steamed = models.BooleanField(default=False)
    Low_fat = models.BooleanField(default=False)
    Prepared_Meals = models.BooleanField(default=False)
    Meats = models.BooleanField(default=False)
    Sweets = models.BooleanField(default=False)
    Beans_and_Lentils = models.BooleanField(default=False)
    Snacks = models.BooleanField(default=False)
    Grains_and_Pasta = models.BooleanField(default=False)
    Nuts_and_Seeds = models.BooleanField(default=False)
    Fruits = models.BooleanField(default=False)
    Beverages = models.BooleanField(default=False)
    Fish = models.BooleanField(default=False)
    Baked_Foods = models.BooleanField(default=False)
    Breakfast_Cereals = models.BooleanField(default=False)
    Vegetables = models.BooleanField(default=False)
    Spices_and_Herbs = models.BooleanField(default=False)
    Soups_and_Sauces = models.BooleanField(default=False)
    Dairy_and_Egg_Products = models.BooleanField(default=False)
    Fats_and_Oils = models.BooleanField(default=False)
    Lactose_Intolerance = models.BooleanField(default=False)
    stones = models.BooleanField(default=False)
    reflux = models.BooleanField(default=False)
    Spices = models.BooleanField(default=False)
    Seafood = models.BooleanField(default=False)
    Poultry = models.BooleanField(default=False)
    Herbs = models.BooleanField(default=False)
    Processed_Meats = models.BooleanField(default=False)
    Legumes = models.BooleanField(default=False)
    Dairy = models.BooleanField(default=False)
    Nuts = models.BooleanField(default=False)
    Mixed_Dishes = models.BooleanField(default=False)
    Sauces = models.BooleanField(default=False)
    Oils = models.BooleanField(default=False)
    Grains = models.BooleanField(default=False)
    Seeds = models.BooleanField(default=False)
    Pastries = models.BooleanField(default=False)
    Spreads = models.BooleanField(default=False)
    Eggs = models.BooleanField(default=False)
    Prebiotic = models.BooleanField(default=False)
    Probiotic = models.BooleanField(default=False)
    Low_Inflammatory = models.BooleanField(default=False)
    Diverse = models.BooleanField(default=False)
    Polyphenol_Rich = models.BooleanField(default=False)
    Stews = models.BooleanField(default=False)
    Mixed_Rice = models.BooleanField(default=False)
    Soups_and_Aashes = models.BooleanField(default=False)
    Traditional_Foods = models.BooleanField(default=False)
    International_Foods = models.BooleanField(default=False)
    Vegetarian_Foods = models.BooleanField(default=False)
    Simple_Dishes = models.BooleanField(default=False)
    Main_Salads = models.BooleanField(default=False)
    Fast_Foods = models.BooleanField(default=False)

    FODMAP_Classification = models.IntegerField(default=0)
    # Allergens
    Allergens_Peanut = models.BooleanField(default=False)
    Allergens_Wheat = models.BooleanField(default=False)
    Allergens_Shrimp = models.BooleanField(default=False)
    Allergens_Mushroom = models.BooleanField(default=False)
    Allergens_Eggplant = models.BooleanField(default=False)
    Allergens_Dried_fruits = models.BooleanField(default=False)
    Allergens_Gluten = models.BooleanField(default=False)
    Allergens_Fish = models.BooleanField(default=False)
    Allergens_Sesame = models.BooleanField(default=False)
    Allergens_Shellfish = models.BooleanField(default=False)
    Allergens_Nuts = models.BooleanField(default=False)
    Allergens_Soy = models.BooleanField(default=False)
    Allergens_Eggs = models.BooleanField(default=False)
    Allergens_Peanuts = models.BooleanField(default=False)
    Allergens_Dairy = models.BooleanField(default=False)
    Allergens_Legumes = models.BooleanField(default=False)
    Allergens_Depends_on_filling_ingredients = models.BooleanField(default=False)
    Allergens_Milk = models.BooleanField(default=False)
    # Diets
    Diet_Low_Sugar = models.BooleanField(default=False)
    Diet_Mediterranean = models.BooleanField(default=False)
    Diet_High_Protein = models.BooleanField(default=False)
    Whole_Foods = models.BooleanField(default=False)
    Diet_Low_Fat = models.BooleanField(default=False)
    Diet_Low_Calorie = models.BooleanField(default=False)
    Diet_Anti_Inflammatory = models.BooleanField(default=False)
    Diet_Paleo = models.BooleanField(default=False)
    Diet_Diabetic = models.BooleanField(default=False)
    Diet_Low_Carb = models.BooleanField(default=False)
    Diet_Bodybuilding = models.BooleanField(default=False)
    Diet_Vegan = models.BooleanField(default=False)
    Diet_Vegetarian = models.BooleanField(default=False)
    Diet_Plant_Based = models.BooleanField(default=False)
    Diet_Gluten_Free = models.BooleanField(default=False)
    Diet_Keto = models.BooleanField(default=False)
    # Other Tags
    bread_rice_tag = models.TextField(default="")
    cooked_rice = models.BooleanField(default=False)
    main_dish = models.BooleanField(default=False)
    GL_category = models.IntegerField(default=0)
    GL_value = models.IntegerField(default=0)
    GI_category = models.IntegerField(default=0)
    GI_value = models.IntegerField(default=0)
    # Diseases
    Disease_Kidney = models.BooleanField(default=False)
    Disease_Kidney_stones = models.BooleanField(default=False)
    Disease_Acid_reflux = models.BooleanField(default=False)
    Disease_Gout = models.BooleanField(default=False)
    Disease_Celiac = models.BooleanField(default=False)
    Disease_IBS = models.BooleanField(default=False)
    Disease_Diverticulitis = models.BooleanField(default=False)
    Disease_Hypothyroidism = models.BooleanField(default=False)
    Disease_Gastroesophageal_Reflux = models.BooleanField(default=False)
    Disease_Foodborne_Illness = models.BooleanField(default=False)
    Disease_High_blood_pressure = models.BooleanField(default=False)
    Disease_High_cholesterol = models.BooleanField(default=False)
    Disease_Diabetes = models.BooleanField(default=False)
    Disease_Hypercholesterolemia = models.BooleanField(default=False)
    Disease_G6PD_Deficiency = models.BooleanField(default=False)
    Disease_Constipation = models.BooleanField(default=False)
    Disease_Gallbladder = models.BooleanField(default=False)
    Disease_Latex_Fruit_Syndrome = models.BooleanField(default=False)
    Disease_Obesity = models.BooleanField(default=False)
    Disease_IBD = models.BooleanField(default=False)
    Disease_NAFLD = models.BooleanField(default=False)
    Disease_Anemia = models.BooleanField(default=False)
    Disease_MS = models.BooleanField(default=False)

    def __str__(self):
        return self.FA_Name

    def save(self, *args, **kwargs):
        super().save(*args, **kwargs)
        cache.delete('all_foods')  # Clear cache whenever a Food entry is saved

    Disease_IBD = models.BooleanField(default=False)
    Disease_NAFLD = models.BooleanField(default=False)
    Disease_Anemia = models.BooleanField(default=False)
    Disease_MS = models.BooleanField(default=False)
class FoodCard(models.Model):
    FA_Name = models.TextField(default="نامعلوم")
    EN_Name = models.TextField(default="Unknown")
    main_dish_code = models.IntegerField(default=-1)
    creator = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, blank=True, related_name="created_foodcards")
    edited_by = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, blank=True, default="", related_name="edited_foodcards")
    created_at = models.DateTimeField(default=datetime.now)
    edited_at = models.DateTimeField(null=True, blank=True)
    Calories = models.FloatField(default=0, db_index=False)
    carb_ratio = models.FloatField(default=0, db_index=False)
    fat_ratio = models.FloatField(default=0, db_index=False)
    protein_ratio = models.FloatField(default=0, db_index=False)
    recipe_description = models.TextField(default="")
    recipe_image_link = models.TextField(default="")
    is_keto = models.BooleanField(default=False, blank=False, db_index=False)
    is_med = models.BooleanField(default=False, blank=False, db_index=False)
    is_conditional_med = models.BooleanField(default=False, blank=False, db_index=False)
    # is_dash = models.BooleanField(default=False, blank=False, db_index=False)
    # is_lowgi = models.BooleanField(default=False, blank=False, db_index=False)
    is_lowcarb = models.BooleanField(default=False, blank=False, db_index=False)
    is_mediumcarb = models.BooleanField(default=False, blank=False, db_index=False)
    is_highcarb = models.BooleanField(default=False, blank=False, db_index=False)
    is_highprotein = models.BooleanField(default=False, blank=False, db_index=False)
    is_lowprotein = models.BooleanField(default=False, blank=False, db_index=False)
    is_mediumprotein = models.BooleanField(default=False, blank=False, db_index=False)
    is_mind = models.BooleanField(default=False, blank=False, db_index=False)
    is_lowfat = models.BooleanField(default=False, blank=False, db_index=False)
    is_mediumfat = models.BooleanField(default=False, blank=False, db_index=False)
    is_highfat = models.BooleanField(default=False, blank=False, db_index=False)
    # is_keto_med = models.BooleanField(default=False, blank=False, db_index=False)
    is_low_fodmap = models.BooleanField(default=False, blank=False, db_index=False)
    is_breakfast = models.BooleanField(default=False, blank=False, db_index=False)
    is_lunch = models.BooleanField(default=False, blank=False, db_index=False)
    is_dinner = models.BooleanField(default=False, blank=False, db_index=False)
    is_snack = models.BooleanField(default=False, blank=False, db_index=False)
    foods = models.JSONField(default=list)

    # New Boolean Fields
    Dessert = models.BooleanField(default=False)
    Added_Meat = models.BooleanField(default=False)
    Ing_Main = models.BooleanField(default=False)
    Beverage = models.BooleanField(default=False)
    Sandwich = models.BooleanField(default=False)
    Condiment = models.BooleanField(default=False)
    Lunch = models.BooleanField(default=False)
    Snack = models.BooleanField(default=False)
    # Bread = models.BooleanField(default=False)
    Entree = models.BooleanField(default=False)
    Soup = models.BooleanField(default=False)
    Side_Dish = models.BooleanField(default=False)
    Appetizer = models.BooleanField(default=False)
    Dinner = models.BooleanField(default=False)
    Pasta = models.BooleanField(default=False)
    Breakfast = models.BooleanField(default=False)
    Salad = models.BooleanField(default=False)
    Ingredient = models.BooleanField(default=False)
    
    # Persian month boolean fields
    is_farvardin = models.BooleanField(default=False, db_index=False)
    is_ordibehesht = models.BooleanField(default=False, db_index=False)  # normalized from "اریبهشت"
    is_khordad = models.BooleanField(default=False, db_index=False)
    is_tir = models.BooleanField(default=False, db_index=False)
    is_mordad = models.BooleanField(default=False, db_index=False)
    is_shahrivar = models.BooleanField(default=False, db_index=False)
    is_mehr = models.BooleanField(default=False, db_index=False)
    is_aban = models.BooleanField(default=False, db_index=False)
    is_azar = models.BooleanField(default=False, db_index=False)
    is_dey = models.BooleanField(default=False, db_index=False)
    is_bahman = models.BooleanField(default=False, db_index=False)
    is_esfand = models.BooleanField(default=False, db_index=False)
    is_coffee = models.BooleanField(default=False, db_index=False)
    is_khorak = models.BooleanField(default=False, db_index=False)
    Bacteria_Acidaminococcus = models.BooleanField(default=False)
    Bacteria_Bacteroidetes = models.BooleanField(default=False)
    Bacteria_Alistipes = models.BooleanField(default=False)
    Bacteria_Bifidobacterium = models.BooleanField(default=False)
    Bacteria_Akkermansia = models.BooleanField(default=False)
    Bacteria_Blautia = models.BooleanField(default=False)
    Bacteria_Fecalibacterium = models.BooleanField(default=False)
    Bacteria_Lawsonibacter_asaccharolyticus = models.BooleanField(default=False)
    Bacteria_Lactobacillus = models.BooleanField(default=False)
    Vegan = models.BooleanField(default=False)
    Umami = models.BooleanField(default=False)
    Slow_cooked = models.BooleanField(default=False)
    Spicy = models.BooleanField(default=False)
    Baked = models.BooleanField(default=False)
    Pickled = models.BooleanField(default=False)
    Low_calorie = models.BooleanField(default=False)
    Grilled = models.BooleanField(default=False)
    Roasted = models.BooleanField(default=False)
    High_protein = models.BooleanField(default=False)
    Fried = models.BooleanField(default=False)
    Sweet = models.BooleanField(default=False)
    Salty = models.BooleanField(default=False)
    Lactose_free = models.BooleanField(default=False)
    Low_carb = models.BooleanField(default=False)
    Cold = models.BooleanField(default=False)
    Vegetarian = models.BooleanField(default=False)
    Sour = models.BooleanField(default=False)
    Sweet_Sour = models.BooleanField(default=False)
    Raw = models.BooleanField(default=False)
    Gluten_free = models.BooleanField(default=False)
    Boiled = models.BooleanField(default=False)
    Steamed = models.BooleanField(default=False)
    Low_fat = models.BooleanField(default=False)
    Prepared_Meals = models.BooleanField(default=False)
    Meats = models.BooleanField(default=False)
    Sweets = models.BooleanField(default=False)
    Beans_and_Lentils = models.BooleanField(default=False)
    Snacks = models.BooleanField(default=False)
    Grains_and_Pasta = models.BooleanField(default=False)
    Nuts_and_Seeds = models.BooleanField(default=False)
    Fruits = models.BooleanField(default=False)
    Beverages = models.BooleanField(default=False)
    Fish = models.BooleanField(default=False)
    Baked_Foods = models.BooleanField(default=False)
    Breakfast_Cereals = models.BooleanField(default=False)
    Vegetables = models.BooleanField(default=False)
    Spices_and_Herbs = models.BooleanField(default=False)
    Soups_and_Sauces = models.BooleanField(default=False)
    Dairy_and_Egg_Products = models.BooleanField(default=False)
    Fats_and_Oils = models.BooleanField(default=False)
    Lactose_Intolerance = models.BooleanField(default=False)
    stones = models.BooleanField(default=False)
    reflux = models.BooleanField(default=False)
    Spices = models.BooleanField(default=False)
    Seafood = models.BooleanField(default=False)
    Poultry = models.BooleanField(default=False)
    Herbs = models.BooleanField(default=False)
    Processed_Meats = models.BooleanField(default=False)
    Legumes = models.BooleanField(default=False)
    Dairy = models.BooleanField(default=False)
    Nuts = models.BooleanField(default=False)
    Mixed_Dishes = models.BooleanField(default=False)
    Sauces = models.BooleanField(default=False)
    Oils = models.BooleanField(default=False)
    Grains = models.BooleanField(default=False)
    Seeds = models.BooleanField(default=False)
    Pastries = models.BooleanField(default=False)
    Spreads = models.BooleanField(default=False)
    Eggs = models.BooleanField(default=False)
    Prebiotic = models.BooleanField(default=False)
    Probiotic = models.BooleanField(default=False)
    Low_Inflammatory = models.BooleanField(default=False)
    Diverse = models.BooleanField(default=False)
    Polyphenol_Rich = models.BooleanField(default=False)
    Stews = models.BooleanField(default=False)
    Mixed_Rice = models.BooleanField(default=False)
    Soups_and_Aashes = models.BooleanField(default=False)
    Traditional_Foods = models.BooleanField(default=False)
    International_Foods = models.BooleanField(default=False)
    Vegetarian_Foods = models.BooleanField(default=False)
    Simple_Dishes = models.BooleanField(default=False)
    Main_Salads = models.BooleanField(default=False)
    Fast_Foods = models.BooleanField(default=False)

    FODMAP_Classification = models.IntegerField(default=0)

    # Allergens
    AL_Dairy_Butter = models.BooleanField(default=False)
    AL_Dairy_Cheese = models.BooleanField(default=False)
    AL_Dairy_Coconut = models.BooleanField(default=False)
    AL_Dairy_CoconutMilk = models.BooleanField(default=False)
    AL_Dairy_Cream = models.BooleanField(default=False)
    AL_Dairy_CreamCheese = models.BooleanField(default=False)
    AL_Dairy_IceCream = models.BooleanField(default=False)
    AL_Dairy_Mayonnaise = models.BooleanField(default=False)
    AL_Dairy_Milk = models.BooleanField(default=False)
    AL_Dairy_Milk_Chocolate = models.BooleanField(default=False)
    AL_Dairy_Whey = models.BooleanField(default=False)
    AL_Dairy_Yogurt = models.BooleanField(default=False)
    AL_Egg = models.BooleanField(default=False)
    AL_Eggs_Egg = models.BooleanField(default=False)
    AL_Eggs_Mayonnaise = models.BooleanField(default=False)
    AL_Fish_Crab = models.BooleanField(default=False)
    AL_Fish_Fish = models.BooleanField(default=False)
    AL_Fish_Shrimp = models.BooleanField(default=False)
    AL_Fish_Tuna = models.BooleanField(default=False)
    AL_Grain_Barley = models.BooleanField(default=False)
    AL_Grain_Bean = models.BooleanField(default=False)
    AL_Grain_Buckwheat = models.BooleanField(default=False)
    AL_Grain_Chickpea = models.BooleanField(default=False)
    AL_Grain_Corn = models.BooleanField(default=False)
    AL_Grain_Lentil = models.BooleanField(default=False)
    AL_Grain_Mung = models.BooleanField(default=False)
    AL_Grain_Oat = models.BooleanField(default=False)
    AL_Grain_Oats = models.BooleanField(default=False)
    AL_Grain_Quinoa = models.BooleanField(default=False)
    AL_Grain_Rice = models.BooleanField(default=False)
    AL_Grain_Rye = models.BooleanField(default=False)
    AL_Grain_Soy = models.BooleanField(default=False)
    AL_Grain_Unspecified = models.BooleanField(default=False)
    AL_Grain_Wheat = models.BooleanField(default=False)
    AL_Grains_Wheat = models.BooleanField(default=False)
    AL_Legume_Beans = models.BooleanField(default=False)
    AL_Legume_Lentil = models.BooleanField(default=False)
    AL_Legume_Pea = models.BooleanField(default=False)
    AL_Legume_RedBean = models.BooleanField(default=False)
    AL_Legume_Soy = models.BooleanField(default=False)
    AL_Legume_Split_Peas = models.BooleanField(default=False)
    AL_Legume_SplitPea = models.BooleanField(default=False)
    AL_Meat_Bacon = models.BooleanField(default=False)
    AL_Meat_Beef = models.BooleanField(default=False)
    AL_Meat_Chicken = models.BooleanField(default=False)
    AL_Meat_Ham = models.BooleanField(default=False)
    AL_Meat_Lamb = models.BooleanField(default=False)
    AL_Meat_Pepperoni = models.BooleanField(default=False)
    AL_Meat_Pork = models.BooleanField(default=False)
    AL_Meat_Processed = models.BooleanField(default=False)
    AL_Meat_Sausage = models.BooleanField(default=False)
    AL_Meat_Turkey = models.BooleanField(default=False)
    AL_Meat_Unspecified = models.BooleanField(default=False)
    AL_Meat_Varies = models.BooleanField(default=False)
    AL_Meat_Various = models.BooleanField(default=False)
    AL_Nuts_Almond = models.BooleanField(default=False)
    AL_Nuts_Coconut = models.BooleanField(default=False)
    AL_Nuts_Hazelnut = models.BooleanField(default=False)
    AL_Nuts_Nuts = models.BooleanField(default=False)
    AL_Nuts_Peanut = models.BooleanField(default=False)
    AL_Nuts_Pine_Nuts = models.BooleanField(default=False)
    AL_Nuts_Pistachio = models.BooleanField(default=False)
    AL_Nuts_Sesame = models.BooleanField(default=False)
    AL_Nuts_Unspecified = models.BooleanField(default=False)
    AL_Nuts_Varies = models.BooleanField(default=False)
    AL_Nuts_Various = models.BooleanField(default=False)
    AL_Nuts_Walnut = models.BooleanField(default=False)
    AL_Poultry_Chicken = models.BooleanField(default=False)
    AL_Seeds_Chia = models.BooleanField(default=False)
    AL_Shellfish_Shrimp = models.BooleanField(default=False)
    AL_Dairy_Cream_Cheese = models.BooleanField(default=False)
    AL_Dairy_Creamer = models.BooleanField(default=False)
    AL_Dairy_Goat_Milk = models.BooleanField(default=False)
    AL_Dairy_Kashk = models.BooleanField(default=False)
    AL_Dairy_Kefir = models.BooleanField(default=False)
    AL_Eggs_Whole = models.BooleanField(default=False)
    AL_Eggs_Whole_Eggs = models.BooleanField(default=False)
    AL_Eggs_Yolk = models.BooleanField(default=False)
    AL_Fish_Anchovy = models.BooleanField(default=False)
    AL_Fish_Carp = models.BooleanField(default=False)
    AL_Fish_Catfish = models.BooleanField(default=False)
    AL_Fish_Caviar = models.BooleanField(default=False)
    AL_Fish_Croaker = models.BooleanField(default=False)
    AL_Fish_Flatfish = models.BooleanField(default=False)
    AL_Fish_Flounder = models.BooleanField(default=False)
    AL_Fish_Grouper = models.BooleanField(default=False)
    AL_Fish_Herring = models.BooleanField(default=False)
    AL_Fish_KingMackerel = models.BooleanField(default=False)
    AL_Fish_Mackerel = models.BooleanField(default=False)
    AL_Fish_Mullet = models.BooleanField(default=False)
    AL_Fish_Pike = models.BooleanField(default=False)
    AL_Fish_Porgy = models.BooleanField(default=False)
    AL_Fish_Roe = models.BooleanField(default=False)
    AL_Fish_Salmon = models.BooleanField(default=False)
    AL_Fish_Sardine = models.BooleanField(default=False)
    AL_Fish_Shark = models.BooleanField(default=False)
    AL_Fish_Snapper = models.BooleanField(default=False)
    AL_Fish_Swordfish = models.BooleanField(default=False)
    AL_Fish_Tilapia = models.BooleanField(default=False)
    AL_Fish_Trout = models.BooleanField(default=False)
    AL_Fish_Unspecified = models.BooleanField(default=False)
    AL_Fish_Whitefish = models.BooleanField(default=False)
    AL_Grain_Bulgur = models.BooleanField(default=False)
    AL_Grain_Cereal = models.BooleanField(default=False)
    AL_Grain_Millet = models.BooleanField(default=False)
    AL_Grain_Pea = models.BooleanField(default=False)
    AL_Grain_SplitPea = models.BooleanField(default=False)
    AL_Grains_Barley = models.BooleanField(default=False)
    AL_Grains_Brown_Rice = models.BooleanField(default=False)
    AL_Grains_Bulgur = models.BooleanField(default=False)
    AL_Grains_Chickpea = models.BooleanField(default=False)
    AL_Grains_Corn = models.BooleanField(default=False)
    AL_Grains_Millet = models.BooleanField(default=False)
    AL_Grains_Oats = models.BooleanField(default=False)
    AL_Grains_Rice = models.BooleanField(default=False)
    AL_Legume_Bean = models.BooleanField(default=False)
    AL_Legume_Chickpea = models.BooleanField(default=False)
    AL_Legume_Fava = models.BooleanField(default=False)
    AL_Legume_FavaBean = models.BooleanField(default=False)
    AL_Legume_Peas = models.BooleanField(default=False)
    AL_Legume_Unspecified = models.BooleanField(default=False)
    AL_Meat_Camel = models.BooleanField(default=False)
    AL_Meat_Duck = models.BooleanField(default=False)
    AL_Meat_Fat = models.BooleanField(default=False)
    AL_Meat_Fish = models.BooleanField(default=False)
    AL_Meat_Goose = models.BooleanField(default=False)
    AL_Meat_Ground = models.BooleanField(default=False)
    AL_Meat_Kidney = models.BooleanField(default=False)
    AL_Meat_Offal = models.BooleanField(default=False)
    AL_Meat_Organ = models.BooleanField(default=False)
    AL_Meat_Ostrich = models.BooleanField(default=False)
    AL_Meat_Pigeon = models.BooleanField(default=False)
    AL_Meat_Quail = models.BooleanField(default=False)
    AL_Meat_Sheep = models.BooleanField(default=False)
    AL_Meat_Veal = models.BooleanField(default=False)
    AL_Nuts_Acorn = models.BooleanField(default=False)
    AL_Nuts_Almonds = models.BooleanField(default=False)
    AL_Nuts_Cashew = models.BooleanField(default=False)
    AL_Nuts_Chestnut = models.BooleanField(default=False)
    AL_Nuts_Hazelnuts = models.BooleanField(default=False)
    AL_Nuts_Hemp = models.BooleanField(default=False)
    AL_Nuts_Mixed = models.BooleanField(default=False)
    AL_Nuts_Nutmeg = models.BooleanField(default=False)
    AL_Nuts_Peanuts = models.BooleanField(default=False)
    AL_Nuts_Pecan = models.BooleanField(default=False)
    AL_Other_Sesame = models.BooleanField(default=False)
    AL_Poultry_Duck = models.BooleanField(default=False)
    AL_Poultry_Offal = models.BooleanField(default=False)
    AL_Seafood_Fish = models.BooleanField(default=False)
    AL_Seeds_Cottonseed = models.BooleanField(default=False)
    AL_Seeds_Flax = models.BooleanField(default=False)
    AL_Seeds_Flaxseed = models.BooleanField(default=False)
    AL_Seeds_Poppy = models.BooleanField(default=False)
    AL_Seeds_Pumpkin = models.BooleanField(default=False)
    AL_Seeds_Safflower = models.BooleanField(default=False)
    AL_Seeds_Sesame = models.BooleanField(default=False)
    AL_Seeds_Sunflower = models.BooleanField(default=False)
    AL_Seeds_Watermelon = models.BooleanField(default=False)
    AL_Sesame_Sesame = models.BooleanField(default=False)
    AL_Soy_Soy = models.BooleanField(default=False)
    AL_Soy_SoySauce = models.BooleanField(default=False)
    AL_Soy_Soybean = models.BooleanField(default=False)
    AL_Soy_Soybeans = models.BooleanField(default=False)
    Allergens_Eggplant = models.BooleanField(default=False)
    Allergens_Mushroom = models.BooleanField(default=False)
    Allergens_Dried_fruits = models.BooleanField(default=False)


# Diets
    Diet_Low_Sugar = models.BooleanField(default=False)
    Diet_Mediterranean = models.BooleanField(default=False)
    Diet_High_Protein = models.BooleanField(default=False)
    Whole_Foods = models.BooleanField(default=False)
    Diet_Low_Fat = models.BooleanField(default=False)
    Diet_Low_Calorie = models.BooleanField(default=False)
    Diet_Anti_Inflammatory = models.BooleanField(default=False)
    Diet_Paleo = models.BooleanField(default=False)
    Diet_Diabetic = models.BooleanField(default=False)
    Diet_Low_Carb = models.BooleanField(default=False)
    Diet_Bodybuilding = models.BooleanField(default=False)
    Diet_Vegan = models.BooleanField(default=False)
    Diet_Vegetarian = models.BooleanField(default=False)
    Diet_Plant_Based = models.BooleanField(default=False)
    Diet_Gluten_Free = models.BooleanField(default=False)
    Diet_Keto = models.BooleanField(default=False)

    # Other Tags
    bread_rice_tag = models.TextField(default="")
    cooked_rice = models.BooleanField(default=False)
    main_dish = models.BooleanField(default=False)
    GL_category = models.IntegerField(default=0)
    GL_value = models.IntegerField(default=0)
    GI_category = models.IntegerField(default=0)
    GI_value = models.IntegerField(default=0)

    # Diseases
    Disease_Kidney = models.BooleanField(default=False)
    Disease_Kidney_stones = models.BooleanField(default=False)
    Disease_Acid_reflux = models.BooleanField(default=False)
    Disease_Gout = models.BooleanField(default=False)
    Disease_Celiac = models.BooleanField(default=False)
    Disease_IBS = models.BooleanField(default=False)
    Disease_Diverticulitis = models.BooleanField(default=False)
    Disease_Hypothyroidism = models.BooleanField(default=False)
    Disease_Gastroesophageal_Reflux = models.BooleanField(default=False)
    Disease_Foodborne_Illness = models.BooleanField(default=False)
    Disease_High_blood_pressure = models.BooleanField(default=False)
    Disease_High_cholesterol = models.BooleanField(default=False)
    Disease_Diabetes = models.BooleanField(default=False)
    Disease_Hypercholesterolemia = models.BooleanField(default=False)
    Disease_G6PD_Deficiency = models.BooleanField(default=False)
    Disease_Constipation = models.BooleanField(default=False)
    Disease_Gallbladder = models.BooleanField(default=False)
    Disease_Latex_Fruit_Syndrome = models.BooleanField(default=False)
    Disease_Obesity = models.BooleanField(default=False)
    Disease_CVD = models.BooleanField(default=False)
    Disease_IBD = models.BooleanField(default=False)
    Disease_Lactose_Intolerance = models.BooleanField(default=False)
    Disease_NAFLD = models.BooleanField(default=False)
    Disease_Anemia = models.BooleanField(default=False)
    Disease_Osteoporosis = models.BooleanField(default=False)
    Disease_Arthritis = models.BooleanField(default=False)
    Disease_Autoimmune = models.BooleanField(default=False)
    Disease_CancerPrevention = models.BooleanField(default=False)
    Disease_MentalHealth = models.BooleanField(default=False)
    Disease_CVD = models.BooleanField(default=False)
    Disease_PCOS = models.BooleanField(default=False)
    Disease_MS = models.BooleanField(default=False)
    Disease_Cyst = models.BooleanField(default=False)

    # Food Types
    Bread = models.BooleanField(default=False)
    rice = models.BooleanField(default=False)
    salad_main_dish = models.BooleanField(default=False)
    sport = models.BooleanField(default=False)
    
    # Seasonal Tags
    spring = models.BooleanField(default=False)
    summer = models.BooleanField(default=False)
    autumn = models.BooleanField(default=False)
    winter = models.BooleanField(default=False)

    def __str__(self):
        return self.FA_Name or self.EN_Name

    def save(self, *args, **kwargs):
        # (step1) in code yekbar baraye pak kardan tag haye rooye card ha bayad run beshe
        # dietary_tags = [
        #     'is_ketbo', 'is_med', 'is_conditional_med', 'is_lowcarb',
        #     'is_mediumcarb', 'is_highcarb', 'is_highprotein', 'is_mediumprotein', 'is_lowprotein', 'is_mind',
        #     'is_low_fodmap', 'is_lowfat', 'is_mediumfat', 'is_highfat'
        # ]
        # FoodCard.objects.all().update(**{tag: False for tag in dietary_tags})
        super().save(*args, **kwargs)
        cache.delete('all_food_cards')  # Clear cache whenever a FoodCard is saved

    class Meta:
        indexes = [
            models.Index(fields=['FA_Name'], name='foodcard_fa_name_idx'),
            models.Index(fields=['EN_Name'], name='foodcard_en_name_idx'),
            models.Index(fields=['FA_Name', 'EN_Name'], name='foodcard_names_composite_idx'),
            models.Index(fields=['Calories'], name='foodcard_calories_idx'),
            # Index for common boolean filters
            models.Index(fields=['is_breakfast'], name='foodcard_breakfast_idx'),
            models.Index(fields=['is_lunch'], name='foodcard_lunch_idx'),
            models.Index(fields=['is_dinner'], name='foodcard_dinner_idx'),
            models.Index(fields=['is_snack'], name='foodcard_snack_idx'),
        ]


class RequestedFoods(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    food_name = models.TextField(default="")
    time = models.DateTimeField(default=datetime.now, null=True)
    Checked = models.BooleanField(default=False, blank=False)


class Drug(models.Model):  # new
    name = models.TextField(default="-")
    EN_Name = models.TextField(default="-")
    description = models.TextField(default="temp")
    DataBaseNumber = models.IntegerField(null=False)

    def __str__(self):
        return self.name


############ events models #####################

class SugarMeasurement(models.Model):  # new
    time = models.DateTimeField()
    amount = models.IntegerField(default=1)
    point = models.FloatField(default=0)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    meal = models.IntegerField(null=True)
    hypo = models.BooleanField(default=False)
    hyper = models.BooleanField(default=False)

    def __str__(self):
        return "amount:{amount}".format(amount=self.amount)


class Activities_log(models.Model):
    activity = models.ForeignKey(Activity, null=True, on_delete=models.SET_NULL)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    start_date_time = models.DateTimeField(null=True)
    end_date_time = models.DateTimeField(null=True)
    amount = models.IntegerField()
    point = models.FloatField(default=0)
    PALx = models.FloatField(default=0)
    energy = models.FloatField(default=0, null=True)

    # def __str__(self):
    #     return "activity:{activity}".format(activity=self.activity.name)


class Walking(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    date = models.DateField(null=True)
    amount = models.IntegerField(default=0)
    steps = models.IntegerField(default=0)
    point = models.FloatField(default=0, null=True)
    PALx = models.FloatField(default=0)
    energy = models.FloatField(default=0, null=True)


class Eating(models.Model):  # new
    amount = models.IntegerField()
    date_time = models.DateTimeField(null=True)
    food = models.ForeignKey(Food, null=True, on_delete=models.SET_NULL)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    meal = models.IntegerField(default=0)
    point = models.FloatField(default=0)

    def __str__(self):
        return "food:{food}, time:{time}".format(food=self.food.FA_Name, time=self.date_time)


class Feeling(models.Model):
    name = models.TextField(default="")
    time = models.DateTimeField(default=datetime.now)
    amount = models.IntegerField(default=0)
    point = models.FloatField(default=0)
    description = models.TextField(null=True)
    user = models.ForeignKey(User, on_delete=models.CASCADE)


class InsulinUsage(models.Model):
    time = models.DateTimeField()
    meal = models.IntegerField(default=-1)  # -1 = not selected
    amount = models.IntegerField(default=0)
    point = models.FloatField(default=0)
    Insulin = models.ForeignKey(Insulin, null=True, on_delete=models.SET_NULL)
    user = models.ForeignKey(User, on_delete=models.CASCADE)

    def __str__(self):
        return "insulin:{insulin}".format(insulin=self.Insulin.name)


class BloodPressure(models.Model):
    time = models.DateTimeField(default=datetime.now, null=True)
    amount = models.FloatField(default=0, null=True)
    over = models.FloatField(default=0, null=True)
    user = models.ForeignKey(User, on_delete=models.CASCADE)

    def __str__(self):
        return "pressure:{amount}".format(amount=self.amount)


class Sleep(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    start = models.FloatField(default=0, null=True)
    end = models.FloatField(default=0, null=True)
    time = models.DateTimeField(default=datetime.now, null=True)
    sleeped = models.TextField(default=0, null=True)    
    is_mid_day_sleep = models.BooleanField(default=False, null=True)


class UserQuestionnaire(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    QandA = models.JSONField(default=dict, blank=True, null=True)

    def __str__(self):
        return f"Questionnaire for {self.user.django_user}"


class MicrobiomeQuestionnaire(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    QandA = models.JSONField(default=dict, blank=True, null=True)

    def __str__(self):
        return f"Questionnaire for {self.user.django_user}"


class LifestyleQuestionnaire(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    QandA = models.JSONField(default=dict, blank=True, null=True)

    def __str__(self):
        return f"Questionnaire for {self.user.django_user}"


@receiver(post_save, sender=UserQuestionnaire)
def update_profile_diet_tags(sender, instance, created, **kwargs):
    if created:  # Only process on new questionnaire creation
        user = instance.user
        qanda = instance.QandA
        try:
            # Fetch CR and gender from profile (assuming they're already set)
            profile = user.profile
            CR = profile.CR
            gender = profile.gender

            # Process the questionnaire to get tags
            result = process_questionnaire(qanda, CR, gender)
            if result is None:
                print(f"Failed to process questionnaire for user {user.django_user}")
                return

            # Save tags and CR to profile
            profile.diet_tags = result["Tags"]
            # profile.CR = result["CR"]  # Update CR if it's recalculated
            profile.save()
            print(f"Updated diet tags for {user.django_user}: {result['Tags']}")

        except Exception as e:
            print(f"Error updating profile tags for {user.django_user}: {e}")


class DrugUsage(models.Model):
    time = models.DateTimeField()
    meal = models.IntegerField(default=0)
    amount = models.IntegerField(default=0)
    point = models.FloatField(default=0)
    drug = models.ForeignKey(Drug, null=True, on_delete=models.SET_NULL)
    user = models.ForeignKey(User, on_delete=models.CASCADE)

    def __str__(self):
        return "drug:{drug}".format(drug=self.drug.name)


class Weights(models.Model):
    time = models.DateTimeField()
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    weight = models.FloatField()
    BMI = models.FloatField()


class AOneC(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    time = models.DateTimeField(default=datetime.now)
    amount = models.FloatField(default=0)


class Points(models.Model):
    date = models.DateField(null=True)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    this_day_Carbohydrates_g = models.TextField(default="&0&0&0&0&0&0&")
    this_day_Carbohydrates_point = models.TextField(default="&0&0&0&0&0&0&")
    this_day_Fat_list_g = models.TextField(default="&0&0&0&0&0&0&")
    this_day_Fat_list_point = models.TextField(default="&0&0&0&0&0&0&")
    this_day_Protein_list_g = models.TextField(default="&0&0&0&0&0&0&")
    this_day_Protein_list_point = models.TextField(default="&0&0&0&0&0&0&")
    this_day_Fiber_list_g = models.TextField(default="&0&0&0&0&0&0&")
    this_day_Fiber_list_point = models.TextField(default="&0&0&0&0&0&0&")
    totalCarboControlPoint = models.FloatField(default=0)
    totalHypoHyperControlPoint = models.FloatField(default=0)
    sugar_point_list_y = models.TextField(default="&0&0&0&0&0&0&")
    totalSugarControlPoint = models.FloatField(default=0)
    HypoHyperNumber = models.IntegerField(default=0)
    PALX = models.FloatField(default=1.1)
    total_points = models.FloatField(default=0)
    insulinCooperation = models.FloatField(default=0)
    drugCooperation = models.FloatField(default=0)
    emotionsCooperation = models.FloatField(default=0)
    sugarCooperation = models.FloatField(default=0)
    foodCooperation = models.FloatField(default=0)
    totalCooperationPoints = models.FloatField(default=0)
    HEI_elements_list = models.TextField(default="&0&0&0&0&0&0&0&0&0&0&0&0&0&0&0&0&0&0&0&0&")  # 14 ---> [0 : 13]
    total_HEI_Point = models.FloatField(default=0)
    total_Calories_Point = models.FloatField(default=0)
    total_Calories_used = models.FloatField(default=0)
    total_Fiber_used = models.FloatField(default=0)
    total_Fiber_Point = models.FloatField(default=0)
    total_Activity_Point = models.FloatField(default=0)
    total_Steps_Point = models.FloatField(default=0)
    total_Steps_goal = models.FloatField(default=5000)
    total_Exercise_Point = models.FloatField(default=0)
    total_Exercise_goal = models.FloatField(default=1)
    total_Engage_Point = models.FloatField(default=0)
    total_Engage_goal = models.FloatField(default=10)
    total_Water_Point = models.FloatField(default=0)
    total_Water_goal = models.FloatField(default=8)
    total_Sleep_Point = models.FloatField(default=0)
    total_Sleep_goal = models.FloatField(default=8)
    this_day_messages = models.IntegerField(default=0)
    this_day_breakfast_Point = models.FloatField(default=0)
    this_day_lunch_Point = models.FloatField(default=0)
    this_day_dinner_Point = models.FloatField(default=0)
    this_day_snack1_Point = models.FloatField(default=0)
    this_day_snack2_Point = models.FloatField(default=0)
    this_day_breakfast_Fiber_Point = models.FloatField(default=0)
    this_day_lunch_Fiber_Point = models.FloatField(default=0)
    this_day_dinner_Fiber_Point = models.FloatField(default=0)
    this_day_snack1_Fiber_Point = models.FloatField(default=0)
    this_day_snack2_Fiber_Point = models.FloatField(default=0)
    this_day_total_diet_Point = models.FloatField(default=0)
    this_day_DataEntry_score = models.FloatField(default=0)


class Admins(models.Model):
    First_Name = models.TextField(default="")
    Last_Name = models.TextField(default="")
    token = models.TextField(null=False)
    UserName = models.TextField(null=False)
    PassWord = models.TextField(null=False)
    foodsEdit = models.BooleanField(default=False)


class Document(models.Model):
    description = models.TextField(blank=True)
    document = models.FileField(upload_to='documents/')
    uploaded_at = models.DateTimeField(auto_now_add=True)


class Notification(models.Model):
    Title = models.TextField(default="No title!")
    Link = models.TextField(default="")
    Summary = models.TextField(default="no summary !")
    datetime = models.DateTimeField(default=datetime.now)


class Message(models.Model):
    user = models.ForeignKey(User, on_delete=models.SET_NULL, null=True)
    employee = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True)
    time = models.DateTimeField(default=datetime.now)
    response_time = models.DateTimeField(null=True, default=None)
    title = models.TextField(default="")
    request_text = models.TextField(default="")
    response_text = models.TextField(default="")
    number_of_fetchs = models.IntegerField(default=0)
    is_answered = models.BooleanField(default=False)
    direction = models.BooleanField(default=False,
                                    null=True)  # 0-False for (user ---> doctor) and  1-True for (doctor ---> user)
    user_seen = models.BooleanField(default=False, null=True)
    employee_seen = models.BooleanField(default=False, null=True)


class UpdateTable(models.Model):
    DateTime = models.DateTimeField(default=datetime.now)
    Type = models.TextField(default="")
    FileUpdateName = models.TextField(default="")


class AppVersion(models.Model):
    time = models.DateTimeField(default=datetime.now)
    version = models.TextField(default="")
    google_play_link = models.TextField(default="")
    bazar_link = models.TextField(default="")
    site_link = models.TextField(default="")
    description = models.TextField(default="")


class GLP1Usage(models.Model):
    time = models.DateTimeField(datetime.now)
    meal = models.IntegerField(default=0)
    amount = models.IntegerField(default=0)
    point = models.FloatField(default=0)
    glp1 = models.ForeignKey(GLP1, null=True, on_delete=models.SET_NULL)
    user = models.ForeignKey(User, on_delete=models.CASCADE)


class FoodImage(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    food = models.ForeignKey(Food, null=True, on_delete=models.SET_NULL)
    time = models.DateTimeField(default=datetime.now, null=True)
    meal = models.IntegerField(default=None, null=True)
    amount = models.FloatField(default=None, null=True)
    is_processed = models.BooleanField(default=False)
    path = models.TextField(default="")


class GlucometerImage(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    time = models.DateTimeField(default=datetime.now, null=True)
    meal = models.IntegerField(default=None, null=True)
    amount = models.FloatField(default=None, null=True)
    is_processed = models.BooleanField(default=False)
    path = models.TextField(default="")


class DiabetesInfo(models.Model):
    DBNumber = models.IntegerField(null=True)
    NumberOfFetched = models.IntegerField(null=True)
    info = models.TextField(null=True)
    Audience = models.TextField(default="all")
    Tags = models.TextField(default="")
    is_Conditional = models.BooleanField(default=False)


class Drugstore(models.Model):
    Name = models.TextField(null=True)
    Province_en = models.TextField(null=True)
    Province_fa = models.TextField(null=True)
    City_en = models.TextField(null=True)
    City_fa = models.TextField(null=True)
    Phone = models.TextField(null=True)
    Address = models.TextField(null=True)


class Analysis(models.Model):  # new
    date = models.DateField(default=date.today, null=True)
    user_counter = models.IntegerField(null=True)
    registered_user_counter = models.IntegerField(null=True)
    active_user_counter = models.IntegerField(null=True)
    insulin_logs = models.IntegerField(null=True)
    activities_logs = models.IntegerField(null=True)
    drug_logs = models.IntegerField(null=True)
    sugar_logs = models.IntegerField(null=True)
    walking_logs = models.IntegerField(null=True)
    feeling_logs = models.IntegerField(null=True)
    message_logs = models.IntegerField(null=True)
    eating_logs = models.IntegerField(null=True)
    pressure_logs = models.IntegerField(null=True)
    sumation_logs = models.IntegerField(null=True)
    active_in_30_days_ago = models.IntegerField(null=True)
    today_active_user_list = models.TextField(null=True)
    today_active_user_transaction_list = models.TextField(null=True)
    new_users = models.IntegerField(null=True)
    api_monitoring = models.JSONField(null=True, default=dict)


class Transaction(models.Model):
    date = models.DateField(default=date.today, null=True)
    error_transactions = models.IntegerField(default=0)
    all_transactions = models.IntegerField(default=0)


class Gift(models.Model):
    created_at = models.DateTimeField(default=datetime.now)
    name = models.TextField(null=True)
    user = models.ForeignKey(to=User, on_delete=models.SET_NULL, null=True)
    value = models.IntegerField(default=0)
    code = models.TextField(null=True)
    description = models.TextField(null=True)
    link = models.TextField(null=True)

    def __str__(self):
        return "name:{name}, value:{value}".format(name=self.name, value=self.value)


class InsulinTypeSelection(models.Model):
    start = models.DateTimeField(default=datetime.now)
    end = models.DateTimeField(default=None, null=True)
    user = models.ForeignKey(to=User, on_delete=models.SET_NULL, null=True)
    insulin = models.ForeignKey(to=Insulin, on_delete=models.SET_NULL, null=True)

    def __str__(self):
        if self.insulin is not None:
            return "insulin type: {insulin}, from :{start}, to:{end}".format(insulin=self.insulin.name,
                                                                             start=self.start, end=self.end)
        else:
            return "from :{start}, to:{end}".format(start=self.start, end=self.end)

    def finish(self):
        self.end = datetime.now()
        self.save()

    @property
    def j_start(self):
        dt = jdatetime.datetime.fromgregorian(datetime=self.start)
        return "{d} {m} {y}".format(y=dt.year, m=dt.j_months_fa[dt.month - 1], d=dt.day)

    @property
    def j_end(self):
        if self.end is None:
            return "الان"
        else:
            dt = jdatetime.datetime.fromgregorian(datetime=self.end)
            return "{d} {m} {y}".format(y=dt.year, m=dt.j_months_fa[dt.month - 1], d=dt.day)


def normalize_unit(unit):
    """
        Converts short English units to Farsi or more readable strings.
        Adjust the mappings as needed.
        """
    unit_mappings = {
        'tsp': 'قاشق چای خوری',
        'g': 'گرم',
        'Kg': 'کیلوگرم',
        'unit': 'واحد',
        'Tbsp/قاشق غذا خوری': 'قاشق غذا خوری',
        'tbsp': 'قاشق غذا خوری',
        # Add more if needed
    }
    return unit_mappings.get(unit, unit)


class Diet(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, null=True)
    creator = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, blank=True)
    created_at = models.DateTimeField(default=datetime.now)
    from_date = models.DateField(default=date.today)
    diet_json = models.JSONField(default=dict, null=True, blank=True)  # 14 options per meal for the month
    eaten_foods = models.JSONField(default=dict, null=True, blank=True)  # 31 days of tracking

    @property
    def html_table(self):
        data = self.diet_json.get("week1", {})  # Access week1, default to empty dict if missing

        html = """<table class="table text-right" style="direction: rtl; font-size: 18px;">
                  <thead>
                  <tr>
                      <th></th>
                      <th>انتخاب اول</th>
                      <th>انتخاب دوم</th>
                      <th>انتخاب سوم</th>
                      <th>انتخاب چهارم</th>
                      <th>انتخاب پنجم</th>
                      <th>انتخاب ششم</th>
                      <th>انتخاب هفتم</th>
                      <th>انتخاب هشتم</th>
                      <th>انتخاب نهم</th>
                      <th>انتخاب دهم</th>
                      <th>انتخاب یازدهم</th>
                      <th>انتخاب دوازدهم</th>
                      <th>انتخاب سیزدهم</th>
                      <th>انتخاب چهاردهم</th>
                  </tr>
                  </thead>
                  <tbody>
                  <tr>
                      <td style="font-weight: bold">صبحانه</td>
                      {}
                  </tr>
                  <tr>
                      <td style="font-weight: bold">ناهار</td>
                      {}
                  </tr>
                  <tr>
                      <td style="font-weight: bold">شام</td>
                      {}
                  </tr>
                  <tr>
                      <td style="font-weight: bold">میان وعده اول</td>
                      {}
                  </tr>
                  <tr>
                      <td style="font-weight: bold">میان وعده دوم</td>
                      {}
                  </tr>
              </table>"""

        meal_mapping = {
            "breakfast": "صبحانه",
            "lunch": "ناهار",
            "dinner": "شام",
            "snack1": "میان وعده اول",
            "snack2": "میان وعده دوم"
        }
        all_td_list = []

        for meal in meal_mapping.keys():
            tds = ""
            meal_data = data.get(meal, [])  # Get meal array from week1, default to empty list
            for i in range(min(14, len(meal_data))):
                model = meal_data[i]  # This is a sub-array like [mealEntry] or []
                td = '<td style="font-size: 14px;">'
                if model and len(model) > 0:  # Check if sub-array has a meal entry
                    food = model[0]  # Take the first (and typically only) entry in the sub-array
                    td += f'<span style="font-size: 16px; color: #009A6E; font-weight: bold">{food.get("title", "")}: </span>'
                    for item in food.get("ingredient", []):
                        try:
                            td += "<br>"
                            td += "{} {} {}".format(
                                item.get("amount_in_home_unit", ""),
                                item.get("home_unit", ""),
                                item.get("name", "")
                            )
                        except:
                            pass
                td += "</td>\n"
                tds += td
            for _ in range(len(meal_data), 14):  # Pad with empty cells if less than 14 options
                tds += '<td style="font-size: 14px;">—</td>\n'
            all_td_list.append(tds)

        return html.format(*all_td_list)

    @property
    def generate_html_table(self):
        data = self.eaten_foods

        html_template = """
        <div class="table-responsive">
          <table class="table diet-table text-right" style="direction: rtl; font-size: 16px;">
            <thead>
              <tr>
                <th></th>
                <th>روز ۱</th><th>روز ۲</th><th>روز ۳</th><th>روز ۴</th><th>روز ۵</th><th>روز ۶</th><th>روز ۷</th>
                <th>روز ۸</th><th>روز ۹</th><th>روز ۱۰</th><th>روز ۱۱</th><th>روز ۱۲</th><th>روز ۱۳</th><th>روز ۱۴</th>
                <th>روز ۱۵</th><th>روز ۱۶</th><th>روز ۱۷</th><th>روز ۱۸</th><th>روز ۱۹</th><th>روز ۲۰</th><th>روز ۲۱</th>
                <th>روز ۲۲</th><th>روز ۲۳</th><th>روز ۲۴</th><th>روز ۲۵</th><th>روز ۲۶</th><th>روز ۲۷</th><th>روز ۲۸</th>
                <th>روز ۲۹</th><th>روز ۳۰</th><th>روز ۳۱</th>
              </tr>
            </thead>
            <tbody>
              <tr>
                <td style="font-weight: bold">صبحانه</td>
                {}
              </tr>
              <tr>
                <td style="font-weight: bold">ناهار</td>
                {}
              </tr>
              <tr>
                <td style="font-weight: bold">شام</td>
                {}
              </tr>
              <tr>
                <td style="font-weight: bold">میان وعده اول</td>
                {}
              </tr>
              <tr>
                <td style="font-weight: bold">میان وعده دوم</td>
                {}
              </tr>
            </tbody>
          </table>
        </div>
        """

        meal_order = ["breakfast", "lunch", "dinner", "snack1", "snack2"]
        all_tds = []

        def build_meal_columns(meal_key):
            meal_dict = data.get(meal_key, {})
            tds = ""
            for day_idx in range(1, 32):  # 1-31 days
                day_key = f"day{day_idx}"
                day_data = meal_dict.get(day_key, {})
                planned_ingredients = {}
                eaten_ingredients = {}

                if day_data.get("selected_mode"):
                    for ing in day_data["selected_mode"].get("ingredient", []):
                        fcode = ing.get("food_code")
                        if fcode:
                            planned_ingredients[fcode] = ing

                if day_data.get("eaten"):
                    for ing in day_data["eaten"].get("ingredient", []):
                        fcode = ing.get("food_code")
                        if fcode:
                            eaten_ingredients[fcode] = ing

                planned_html_list = ""
                for ing in planned_ingredients.values():
                    amt = ing.get("amount_in_home_unit") or ing.get("amount_g", 0)
                    amt_str = f"{float(amt):.2f}"
                    unit = normalize_unit(ing.get("home_unit", "g"))
                    name = ing.get("name", "ناشناس")
                    planned_html_list += f"<li>{amt_str} {unit} {name}</li>"

                planned_html = f"""
                  <div class="planned-items">
                    <strong>پیشنهادی:</strong>
                    <ul>{planned_html_list or '<li>—</li>'}</ul>
                  </div>
                """

                eaten_html_list = ""
                all_fcodes = set(planned_ingredients.keys()) | set(eaten_ingredients.keys())
                for fcode in all_fcodes:
                    pitem = planned_ingredients.get(fcode)
                    eitem = eaten_ingredients.get(fcode)
                    if pitem is None and eitem is not None:
                        amt = eitem.get("amount_in_home_unit") or eitem.get("amount_g", 0)
                        amt_str = f"{float(amt):.2f}"
                        unit = normalize_unit(eitem.get("home_unit", "g"))
                        name = eitem.get("name", "ناشناس")
                        eaten_html_list += f"<li class='extra-item'>{amt_str} {unit} {name} (افزوده‌شده)</li>"
                    elif eitem is None and pitem is not None:
                        amt = pitem.get("amount_in_home_unit") or pitem.get("amount_g", 0)
                        amt_str = f"{float(amt):.2f}"
                        unit = normalize_unit(pitem.get("home_unit", "g"))
                        name = pitem.get("name", "ناشناس")
                        eaten_html_list += f"<li class='nonmatch-item'>{amt_str} {unit} {name} (نخورده)</li>"
                    else:
                        planned_amt = float(pitem.get("amount_in_home_unit") or pitem.get("amount_g", 0))
                        eaten_amt = float(eitem.get("amount_in_home_unit") or eitem.get("amount_g", 0))
                        unit = normalize_unit(eitem.get("home_unit", "g"))
                        name = eitem.get("name", "ناشناس")
                        threshold = planned_amt * 0.1
                        css_class = "match-item" if abs(eaten_amt - planned_amt) <= threshold else "partial-item"
                        eaten_html_list += f"<li class='{css_class}'>{eaten_amt:.2f} {unit} {name}</li>"

                eaten_html = f"""
                  <div class="eaten-items">
                    <strong>مصرف‌شده:</strong>
                    <ul>{eaten_html_list or '<li>—</li>'}</ul>
                  </div>
                """

                tds += f"<td>{planned_html}{eaten_html}</td>"
            return tds

        for meal_key in meal_order:
            all_tds.append(build_meal_columns(meal_key))

        return html_template.format(*all_tds)

    def eating(self, day: int, meal: int, model: int, amounts: list, food_codes: list, food_names: list,
               carbohydrates: list, proteins: list, fats: list, calories: list, home_unit: list,
               amount_in_home_unit: list):
        if not 0 <= day <= 30:  # 0-based index for 31 days
            raise ValueError("Day must be between 0 and 30")
        day_str = f"day{day + 1}"
        meals = ["breakfast", "lunch", "dinner", "snack1", "snack2"]
        meal_str = meals[meal]

        # No limit on model selection
        if model != -1:
            if not 0 <= model < 14:
                raise ValueError("Model index must be between 0 and 13 for 14 options")
            selected_model = self.diet_json.get("week1", {}).get(meal_str, [])[model][0]  # Updated to access week1

        # Initialize eaten_foods for 31 days
        if not self.eaten_foods:
            self.eaten_foods = {
                "breakfast": {f"day{i}": {"selected_mode": None, "eaten": {"ingredient": []}} for i in range(1, 32)},
                "lunch": {f"day{i}": {"selected_mode": None, "eaten": {"ingredient": []}} for i in range(1, 32)},
                "dinner": {f"day{i}": {"selected_mode": None, "eaten": {"ingredient": []}} for i in range(1, 32)},
                "snack1": {f"day{i}": {"selected_mode": None, "eaten": {"ingredient": []}} for i in range(1, 32)},
                "snack2": {f"day{i}": {"selected_mode": None, "eaten": {"ingredient": []}} for i in range(1, 32)},
            }

        if model != -1:
            self.eaten_foods[meal_str][day_str]["selected_mode"] = selected_model
        for i in range(len(amounts)):
            item = {
                "food_code": food_codes[i],
                "amount_g": amounts[i],
                "carbohydrate": carbohydrates[i],
                "protein": proteins[i],
                "fat": fats[i],
                "energy": calories[i],
                "name": food_names[i],
                "home_unit": home_unit[i],
                "amount_in_home_unit": amount_in_home_unit[i],
            }
            self.eaten_foods[meal_str][day_str]["eaten"]["ingredient"].append(item)
        self.save()

    @property
    def to_json(self):
        s = dict()
        s['diet'] = self.diet_json
        s['eaten_foods'] = self.eaten_foods
        s['from_date'] = self.from_date
        return s

    def adjust_starting_date(self):
        last_diet = Diet.objects.filter(user=self.user,
                                        from_date__range=((datetime.now() - timedelta(days=31)).date(),
                                                          (datetime.now() - timedelta(days=1)).date())).last()
        if last_diet is None:
            self.from_date = date.today()
        else:
            self.from_date = last_diet.from_date + timedelta(days=31)  # Monthly cycle
        self.save()

    @property
    def jalali_starting_date(self):
        jdt = jdatetime.date.fromgregorian(date=self.from_date)
        return f"{jdt.j_weekdays_fa[jdt.weekday()]} {jdt.day} {jdt.j_months_fa[jdt.month - 1]}"


class DietTemplate(models.Model):
    template_identifier = models.CharField(max_length=255, unique=True, null=True, blank=True)
    creator = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, blank=True)
    created_at = models.DateTimeField(default=datetime.now)
    diet_json = models.JSONField(default=dict, null=True, blank=True)
    diet_name = models.CharField(max_length=255, null=True, blank=True, db_index=True)
    calorie_range = models.IntegerField(default=0, db_index=True)
    company_name = models.CharField(max_length=255, null=True, blank=True, db_index=True)  # For tagging "Nestle"
    city = models.CharField(max_length=255, null=True, blank=True, db_index=True)  # e.g., "Tehran", "Qazvin"
    is_user_template = models.BooleanField(default=False)

    # Bacteria Type
    akkermansia = models.BooleanField(default=False)
    bacteroides = models.BooleanField(default=False)
    bifidobacterium = models.BooleanField(default=False)
    faecalibacterium = models.BooleanField(default=False)
    alistipes = models.BooleanField(default=False)
    eubacterium = models.BooleanField(default=False)
    lactobacillus = models.BooleanField(default=False)
    blautia = models.BooleanField(default=False)

    # Diet Type
    med = models.BooleanField(default=False)
    low_gi = models.BooleanField(default=False)
    fasting = models.BooleanField(default=False)
    keto = models.BooleanField(default=False)
    mind = models.BooleanField(default=False)
    sport = models.BooleanField(default=False)
    liquid_or_pureed_diet = models.BooleanField(default=False)
    eliminate_diet = models.BooleanField(default=False)
    low_fodmap = models.BooleanField(default=False)
    pregnant = models.BooleanField(default=False)
    lactating = models.BooleanField(default=False)

    # Diseases
    fatty_liver = models.BooleanField(default=False)
    diabetes = models.BooleanField(default=False)
    ibs = models.BooleanField(default=False)
    pcos = models.BooleanField(default=False)
    neghres = models.BooleanField(default=False)
    ms = models.BooleanField(default=False)
    obesity_surgery = models.BooleanField(default=False)
    renal = models.BooleanField(default=False)
    cvd = models.BooleanField(default=False)
    migraine = models.BooleanField(default=False)
    cancer = models.BooleanField(default=False)
    nervous_disease = models.BooleanField(default=False)
    gi_disease = models.BooleanField(default=False)
    healthy = models.BooleanField(default=False)

    # New Boolean Fields
    Dessert = models.BooleanField(default=False)
    Added_Meat = models.BooleanField(default=False)
    Ing_Main = models.BooleanField(default=False)
    Beverage = models.BooleanField(default=False)
    Sandwich = models.BooleanField(default=False)
    Condiment = models.BooleanField(default=False)
    Lunch = models.BooleanField(default=False)
    Snack = models.BooleanField(default=False)
    Bread = models.BooleanField(default=False)
    Entree = models.BooleanField(default=False)
    Soup = models.BooleanField(default=False)
    Side_Dish = models.BooleanField(default=False)
    Appetizer = models.BooleanField(default=False)
    Dinner = models.BooleanField(default=False)
    Pasta = models.BooleanField(default=False)
    Breakfast = models.BooleanField(default=False)
    Salad = models.BooleanField(default=False)
    Ingredient = models.BooleanField(default=False)
    Bacteria_Acidaminococcus = models.BooleanField(default=False)
    Bacteria_Lawsonibacter_asaccharolyticus = models.BooleanField(default=False)
    Vegan = models.BooleanField(default=False)
    Umami = models.BooleanField(default=False)
    Slow_cooked = models.BooleanField(default=False)
    Spicy = models.BooleanField(default=False)
    Baked = models.BooleanField(default=False)
    Pickled = models.BooleanField(default=False)
    Low_calorie = models.BooleanField(default=False)
    Grilled = models.BooleanField(default=False)
    Roasted = models.BooleanField(default=False)
    High_protein = models.BooleanField(default=False)
    Fried = models.BooleanField(default=False)
    Sweet = models.BooleanField(default=False)
    Salty = models.BooleanField(default=False)
    Lactose_free = models.BooleanField(default=False)
    Low_carb = models.BooleanField(default=False)
    Cold = models.BooleanField(default=False)
    Vegetarian = models.BooleanField(default=False)
    Sour = models.BooleanField(default=False)
    Sweet_Sour = models.BooleanField(default=False)
    Raw = models.BooleanField(default=False)
    Gluten_free = models.BooleanField(default=False)
    Boiled = models.BooleanField(default=False)
    Steamed = models.BooleanField(default=False)
    Low_fat = models.BooleanField(default=False)
    Prepared_Meals = models.BooleanField(default=False)
    Meats = models.BooleanField(default=False)
    Sweets = models.BooleanField(default=False)
    Beans_and_Lentils = models.BooleanField(default=False)
    Snacks = models.BooleanField(default=False)
    Grains_and_Pasta = models.BooleanField(default=False)
    Nuts_and_Seeds = models.BooleanField(default=False)
    Fruits = models.BooleanField(default=False)
    Beverages = models.BooleanField(default=False)
    Fish = models.BooleanField(default=False)
    Baked_Foods = models.BooleanField(default=False)
    Breakfast_Cereals = models.BooleanField(default=False)
    Vegetables = models.BooleanField(default=False)
    Spices_and_Herbs = models.BooleanField(default=False)
    Soups_and_Sauces = models.BooleanField(default=False)
    Dairy_and_Egg_Products = models.BooleanField(default=False)
    Fats_and_Oils = models.BooleanField(default=False)
    Lactose_Intolerance = models.BooleanField(default=False)
    stones = models.BooleanField(default=False)
    reflux = models.BooleanField(default=False)
    Spices = models.BooleanField(default=False)
    Seafood = models.BooleanField(default=False)
    Poultry = models.BooleanField(default=False)
    Herbs = models.BooleanField(default=False)
    Processed_Meats = models.BooleanField(default=False)
    Legumes = models.BooleanField(default=False)
    Dairy = models.BooleanField(default=False)
    Nuts = models.BooleanField(default=False)
    Mixed_Dishes = models.BooleanField(default=False)
    Sauces = models.BooleanField(default=False)
    Oils = models.BooleanField(default=False)
    Grains = models.BooleanField(default=False)
    Seeds = models.BooleanField(default=False)
    Pastries = models.BooleanField(default=False)
    Spreads = models.BooleanField(default=False)
    Eggs = models.BooleanField(default=False)
    Prebiotic = models.BooleanField(default=False)
    Probiotic = models.BooleanField(default=False)
    Low_Inflammatory = models.BooleanField(default=False)
    Diverse = models.BooleanField(default=False)
    Polyphenol_Rich = models.BooleanField(default=False)
    Stews = models.BooleanField(default=False)
    Mixed_Rice = models.BooleanField(default=False)
    Soups_and_Aashes = models.BooleanField(default=False)
    Traditional_Foods = models.BooleanField(default=False)
    International_Foods = models.BooleanField(default=False)
    Vegetarian_Foods = models.BooleanField(default=False)
    Simple_Dishes = models.BooleanField(default=False)
    Main_Salads = models.BooleanField(default=False)
    Fast_Foods = models.BooleanField(default=False)

    FODMAP_Classification = models.IntegerField(default=0)
    # Allergens
    Allergens_Peanut = models.BooleanField(default=False)
    Allergens_Wheat = models.BooleanField(default=False)
    Allergens_Shrimp = models.BooleanField(default=False)
    Allergens_Mushroom = models.BooleanField(default=False)
    Allergens_Eggplant = models.BooleanField(default=False)
    Allergens_Dried_fruits = models.BooleanField(default=False)
    Allergens_Gluten = models.BooleanField(default=False)
    Allergens_Fish = models.BooleanField(default=False)
    Allergens_Sesame = models.BooleanField(default=False)
    Allergens_Shellfish = models.BooleanField(default=False)
    Allergens_Nuts = models.BooleanField(default=False)
    Allergens_Soy = models.BooleanField(default=False)
    Allergens_Eggs = models.BooleanField(default=False)
    Allergens_Peanuts = models.BooleanField(default=False)
    Allergens_Dairy = models.BooleanField(default=False)
    Allergens_Legumes = models.BooleanField(default=False)
    Allergens_Depends_on_filling_ingredients = models.BooleanField(default=False)
    Allergens_Milk = models.BooleanField(default=False)
    Allergens_Red_meat = models.BooleanField(default=False)
    Allergens_Tomato = models.BooleanField(default=False)
    Allergens_Sea_food = models.BooleanField(default=False)
    Allergens_Fava = models.BooleanField(default=False)
    Allergens_Coffee = models.BooleanField(default=False)
    # Diets
    Diet_Low_Sugar = models.BooleanField(default=False)
    Diet_High_Protein = models.BooleanField(default=False)
    Whole_Foods = models.BooleanField(default=False)
    Diet_Low_Fat = models.BooleanField(default=False)
    Diet_Low_Calorie = models.BooleanField(default=False)
    Diet_Anti_Inflammatory = models.BooleanField(default=False)
    Diet_Paleo = models.BooleanField(default=False)
    Diet_Diabetic = models.BooleanField(default=False)
    Diet_Low_Carb = models.BooleanField(default=False)
    Diet_Bodybuilding = models.BooleanField(default=False)
    Diet_Vegan = models.BooleanField(default=False)
    Diet_Vegetarian = models.BooleanField(default=False)
    Diet_Plant_Based = models.BooleanField(default=False)
    Diet_Gluten_Free = models.BooleanField(default=False)
    # Other Tags
    bread_rice_tag = models.TextField(default="")
    cooked_rice = models.BooleanField(default=False)
    main_dish = models.BooleanField(default=False)
    GL_category = models.IntegerField(default=0)
    GL_value = models.IntegerField(default=0)
    GI_category = models.IntegerField(default=0)
    GI_value = models.IntegerField(default=0)
    # Diseases
    Disease_Kidney = models.BooleanField(default=False)
    Disease_Kidney_stones = models.BooleanField(default=False)
    Disease_Acid_reflux = models.BooleanField(default=False)
    Disease_Gout = models.BooleanField(default=False)
    Disease_Celiac = models.BooleanField(default=False)
    Disease_IBS = models.BooleanField(default=False)
    Disease_Diverticulitis = models.BooleanField(default=False)
    Disease_Hypothyroidism = models.BooleanField(default=False)
    Disease_Gastroesophageal_Reflux = models.BooleanField(default=False)
    Disease_Foodborne_Illness = models.BooleanField(default=False)
    Disease_High_blood_pressure = models.BooleanField(default=False)
    Disease_High_cholesterol = models.BooleanField(default=False)
    Disease_Hypercholesterolemia = models.BooleanField(default=False)
    Disease_G6PD_Deficiency = models.BooleanField(default=False)
    Disease_Constipation = models.BooleanField(default=False)
    Disease_Gallbladder = models.BooleanField(default=False)
    Disease_Latex_Fruit_Syndrome = models.BooleanField(default=False)
    Disease_Obesity = models.BooleanField(default=False)
    Disease_Diarrhea = models.BooleanField(default=False)
    Disease_Cyst = models.BooleanField(default=False)

    # counting tags for meals
    lunch_Stews = models.IntegerField(default=0)
    lunch_Mixed_Rice = models.IntegerField(default=0)
    lunch_Grilled = models.IntegerField(default=0)
    lunch_Soups_and_Aashes = models.IntegerField(default=0)
    lunch_Pasta = models.IntegerField(default=0)
    lunch_Seafood = models.IntegerField(default=0)
    lunch_Traditional_Foods = models.IntegerField(default=0)
    lunch_International_Foods = models.IntegerField(default=0)
    lunch_Vegetarian_Foods = models.IntegerField(default=0)
    lunch_Simple_Dishes = models.IntegerField(default=0)
    lunch_Main_Salads = models.IntegerField(default=0)
    lunch_Fast_Foods = models.IntegerField(default=0)
    dinner_Stews = models.IntegerField(default=0)
    dinner_Mixed_Rice = models.IntegerField(default=0)
    dinner_Grilled = models.IntegerField(default=0)
    dinner_Soups_and_Aashes = models.IntegerField(default=0)
    dinner_Pasta = models.IntegerField(default=0)
    dinner_Seafood = models.IntegerField(default=0)
    dinner_Traditional_Foods = models.IntegerField(default=0)
    dinner_International_Foods = models.IntegerField(default=0)
    dinner_Vegetarian_Foods = models.IntegerField(default=0)
    dinner_Simple_Dishes = models.IntegerField(default=0)
    dinner_Main_Salads = models.IntegerField(default=0)
    dinner_Fast_Foods = models.IntegerField(default=0)

    def __str__(self):
        return self.diet_name

    # the script to run for the ID for templates ----------------------------------------------

    # templates = DietTemplate.objects.all().order_by('id')
    #
    # current_id = 1
    # for template in templates:
    #     template.template_id = str(current_id)
    #     template.save()
    #     current_id += 1

    @property
    def to_json(self):
        s = dict()
        s['diet'] = self.diet_json
        s['eaten_foods'] = self.eaten_foods
        s['from_date'] = self.from_date
        return s


class Water(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    time = models.DateTimeField()
    amount = models.IntegerField(default=1)
    point = models.FloatField(default=0)


class ZiLuckGroup(Group):
    is_public = models.BooleanField(default=0, choices=((False, 'private'), (True, 'public')))
    group_id = models.CharField(max_length=150, blank=True, null=True)
    is_company = models.BooleanField(default=0, choices=((False, 'no_company'), (True, 'company')))
    creator = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, blank=True, related_name='creator')
    admin = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, blank=True, related_name='admin')
    created_at = models.DateTimeField(default=datetime.now, null=True, blank=True)

    @property
    def link(self):
        return 'joining link/{}'.format(self.group_id)

    @property
    def members_number(self):
        # todo : calculate the number of members and return
        return self.user_set.all().count()


class Video(models.Model):
    file_path = models.FileField(upload_to='videos/')
    description = models.TextField(blank=True)


class PDF(models.Model):
    file_path = models.FileField(upload_to='pdfs/')
    description = models.TextField(blank=True)


class Shop(models.Model):
    name = models.CharField(max_length=255)
    image = models.ImageField(null=True, blank=True, upload_to='shop_images/')  # upload_to='shop_images',
    price = models.IntegerField(default=0)
    is_coin = models.BooleanField(default=True, choices=((True, 'coin'), (False, 'no_coin')))
    tag = models.TextField(default="", unique=True)
    description = models.TextField(default="")
    service_type = models.CharField(
        max_length=50,
        choices=(("coaching", "Coaching"), ("content", "Content"), ("plan", "Plan")),
        default="content"
    )
    promotion_percentage = models.IntegerField(default=0)
    promotion_end_date = models.DateTimeField(null=True, blank=True)
    plan_type = models.IntegerField(
        choices=((1, 'Silver'), (2, 'Golden'), (3, 'Platinum'), (4, 'Bronze')),
        null=True, blank=True, help_text="Applicable for plans only"
    )
    duration_months = models.IntegerField(default=0, help_text="Duration in months, for plans only")

    def get_current_price(self):
        if self.promotion_percentage > 0 and self.promotion_end_date and datetime.now() < self.promotion_end_date:
            discount = self.price * (self.promotion_percentage / 100)
            return max(self.price - discount, 0)
        return self.price

    def save(self, *args, **kwargs):
        if ' ' in self.tag:
            raise ValueError("Tags cannot contain spaces.")
        if self.service_type == "plan" and (self.plan_type is None or self.duration_months <= 0):
            raise ValueError("Plans must have a valid plan_type and duration.")
        super().save(*args, **kwargs)


class Purchase(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    shop_item = models.ForeignKey(Shop, on_delete=models.CASCADE)
    purchase_date = models.DateTimeField(default=datetime.now)

    def __str__(self):
        return f"{self.user} purchased {self.shop_item.name} on {self.purchase_date}"


class DietSubscription(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    diet_plan = models.ForeignKey(Shop, on_delete=models.CASCADE)
    start_date = models.DateField(default=datetime.now)
    duration_weeks = models.IntegerField()
    end_date = models.DateField()

    def save(self, *args, **kwargs):
        if not self.end_date:
            self.end_date = self.start_date + timedelta(weeks=self.duration_weeks)
        super().save(*args, **kwargs)


class AppointmentCredit(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    purchase = models.ForeignKey(Purchase, on_delete=models.CASCADE, null=True, blank=True)
    total_credits = models.IntegerField()
    used_credits = models.IntegerField(default=0)

    def available_credits(self):
        return self.total_credits - self.used_credits

    def __str__(self):
        return f"{self.user.django_user} has {self.available_credits()} appointment credits remaining"


class ContentService(models.Model):
    shop_item = models.ForeignKey(Shop, on_delete=models.CASCADE)
    pdf_files = models.ManyToManyField(PDF)
    videos = models.ManyToManyField(Video)


class CoinValue(models.Model):
    price_of_coin = models.FloatField(default=1000)  # Toman
    created_at = models.DateTimeField(default=datetime.now, null=True, blank=True)


def user_directory_path(instance, filename):
    # user_scripts/username/filename
    return os.path.join('user_scripts', instance.django_user.username, filename)


class UserScript(models.Model):
    django_user = models.ForeignKey(DjangoUser, on_delete=models.CASCADE)
    script_file = models.FileField(upload_to=user_directory_path)
    created_at = models.DateTimeField(auto_now_add=True)
    returncode = models.IntegerField(default=None, null=True)
    output = models.TextField(default=None, null=True)

    def __str__(self):
        return self.django_user.username


class ApiRequestLog(models.Model):
    url = models.CharField(max_length=255, verbose_name="API URL")
    response_time = models.FloatField(verbose_name="Response Time")
    timestamp = models.DateTimeField(auto_now_add=True, verbose_name="Timestamp")

    def __str__(self):
        return f"{self.url} - {self.response_time}s"

    @classmethod
    def get_url_stats(cls):
        """
        Calculate stats (avg, max, count) for each unique URL and sort by avg response time descending.
        """
        stats = list(
            cls.objects.values('url').annotate(
                avg=Avg('response_time'),
                max=Max('response_time'),
                count=Count('id')
            ).values('url', 'avg', 'max', 'count')
        )
        # Sort the stats by 'avg' descending
        stats.sort(key=lambda x: x['avg'], reverse=True)
        return {item['url']: {'avg': item['avg'], 'max': item['max'], 'count': item['count']} for item in stats}


class FreeDietPlan(models.Model):
    name = models.CharField(max_length=255)
    phone_number = models.CharField(max_length=50)
    diet_html = models.TextField()  # Stores the generated diet plan (HTML)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.name} - {self.phone_number}"


class NotificationLog(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='notification_logs')
    title = models.CharField(max_length=255)
    body = models.TextField()
    sent_at = models.DateTimeField(default=now)
    status = models.CharField(
        max_length=20,
        choices=(('success', 'Success'), ('failed', 'Failed')),
        default='failed'
    )
    error_message = models.TextField(null=True, blank=True)  # Optional: store failure reason

    def __str__(self):
        return f"{self.user.identifier} - {self.title} ({self.status})"

    class Meta:
        indexes = [
            models.Index(fields=['sent_at']),
            models.Index(fields=['status']),
        ]



class WellnessProfile(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    profile = models.TextField()
    hydration = models.IntegerField(default=0)
    diet = models.IntegerField(default=0)
    dietary_habits = models.IntegerField(default=0)
    sleep = models.IntegerField(default=0)
    exercise = models.IntegerField(default=0)
    mental_wellbeing = models.IntegerField(default=0)
    logging_completeness = models.IntegerField(default=0)
    wellness_score = models.IntegerField(default=0)
    notification_title = models.TextField(default="")
    notification_body = models.TextField(default="")
    recommendation = models.TextField(default="")
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.user.identifier} - {self.created_at}"    


class Category(models.Model):
    name = models.TextField(default="")
    slug = models.SlugField(max_length=255, unique=True, blank=True)
    description = models.TextField(blank=True)
    parent = models.ForeignKey('self', null=True, blank=True, related_name='subcategories', on_delete=models.CASCADE)
    ordering = models.PositiveIntegerField(default=0)
    is_active = models.BooleanField(default=True)
    creator = models.ForeignKey(DjangoUser, on_delete=models.CASCADE, related_name='created_categories')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = ('parent', 'name')
        ordering = ['parent__id', 'ordering', 'name']
        indexes = [
            models.Index(fields=['slug']),
            models.Index(fields=['is_active']),
            models.Index(fields=['creator']),
        ]

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            
            # Check if slug exists and generate unique one
            while Category.objects.filter(slug=slug).exclude(id=self.id).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            
            self.slug = slug
        super().save(*args, **kwargs)

    def __str__(self):
        return self.name

    @property
    def resource_count(self):
        return self.resources.count()

class Resource(models.Model):
    class ResourceType(models.TextChoices):
        VIDEO = 'video', 'Video'
        PDF = 'pdf', 'PDF'
        IMAGE_SERIES = 'image_series', 'Image Series'

    title = models.TextField(default="")
    slug = models.SlugField(max_length=255, unique=True, blank=True)
    description = models.TextField(blank=True)
    category = models.ForeignKey(Category, related_name='resources', on_delete=models.CASCADE)
    type = models.CharField(max_length=20, choices=ResourceType.choices)
    is_published = models.BooleanField(default=False)
    is_featured = models.BooleanField(default=False)
    view_count = models.PositiveIntegerField(default=0)
    ordering = models.PositiveIntegerField(default=0)
    creator = models.ForeignKey(DjangoUser, on_delete=models.CASCADE, related_name='created_resources')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ['category', 'ordering', 'title']
        indexes = [
            models.Index(fields=['slug']),
            models.Index(fields=['type']),
            models.Index(fields=['is_published']),
            models.Index(fields=['is_featured']),
            models.Index(fields=['category']),
        ]

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.title)
            slug = base_slug
            counter = 1
            
            # Check if slug exists and generate unique one
            while Resource.objects.filter(slug=slug).exclude(id=self.id).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            
            self.slug = slug
        super().save(*args, **kwargs)

    def __str__(self):
        return self.title

class VideoResource(models.Model):
    resource = models.OneToOneField(Resource, on_delete=models.CASCADE, related_name='video')
    video_url = models.URLField()
    duration_minutes = models.PositiveIntegerField()
    thumbnail = models.ImageField(upload_to='resource_thumbnails/')

class PDFResource(models.Model):
    resource = models.OneToOneField(Resource, on_delete=models.CASCADE, related_name='pdf')
    file = models.FileField(upload_to='resource_pdfs/')
    file_size = models.PositiveIntegerField()
    page_count = models.PositiveIntegerField()

class ImageSeriesResource(models.Model):
    resource = models.OneToOneField(Resource, on_delete=models.CASCADE, related_name='image_series')

class ImageSeriesItem(models.Model):
    image_series = models.ForeignKey(ImageSeriesResource, on_delete=models.CASCADE, related_name='items')
    image = models.ImageField(upload_to='resource_image_series/')
    caption = models.CharField(max_length=255, blank=True)
    ordering = models.PositiveIntegerField(default=0)

    class Meta:
        ordering = ['ordering', 'id']

class UserResourceInteraction(models.Model):
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    resource = models.ForeignKey(Resource, on_delete=models.CASCADE)
    viewed = models.BooleanField(default=False)
    completed = models.BooleanField(default=False)
    bookmarked = models.BooleanField(default=False)
    liked = models.BooleanField(default=False)
    progress = models.FloatField(default=0)  # 0-100%
    first_viewed_at = models.DateTimeField(null=True, blank=True)
    last_viewed_at = models.DateTimeField(null=True, blank=True)
    completed_at = models.DateTimeField(null=True, blank=True)
    time_spent_seconds = models.PositiveIntegerField(default=0)

    class Meta:
        unique_together = ('user', 'resource')
        indexes = [
            models.Index(fields=['user', 'resource']),
            models.Index(fields=['resource']),
        ]

class UserCategoryProgress(models.Model):
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    category = models.ForeignKey(Category, on_delete=models.CASCADE)
    total_resources = models.PositiveIntegerField(default=0)
    viewed_resources = models.PositiveIntegerField(default=0)
    completed_resources = models.PositiveIntegerField(default=0)
    completion_percent = models.FloatField(default=0)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = ('user', 'category')
        indexes = [
            models.Index(fields=['user', 'category']),
        ]

    def __str__(self):
        return f"{self.user} - {self.category}: {self.completion_percent}%"

class Mood(models.Model):
    MOOD_CHOICES = [
        (1, 'Angry'),
        (2, 'Anxious'),
        (3, 'Calm'),
        (4, 'Grateful'),
        (5, 'Happy'),
        (6, 'Joyful'),
        (7, 'Overwhelmed'),
        (8, 'Sad'),
        (9, 'Stress'),
    ]
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='moods')
    mood = models.PositiveSmallIntegerField(choices=MOOD_CHOICES)
    note = models.TextField(blank=True, null=True)
    created_at = models.DateTimeField(default=datetime.now, blank=True)

    def __str__(self):
        return f"{self.user.identifier} - {self.get_mood_display()} at {self.created_at}"

class Comment(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='comments')
    content = models.TextField(default=None, null=True)
    created_at = models.DateTimeField(default=datetime.now)
    updated_at = models.DateTimeField(auto_now=True)
    is_active = models.BooleanField(default=True)
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return f"Comment by {self.user.identifier} at {self.created_at}"

class BreathWork(models.Model):
    BREATH_WORK_CHOICES = [
        (1, 'Anchor Breath Work'),
        (2, 'Box Breath Work'),
        (3, 'Sleep Breath Work'),
        (4, 'Self Relief Training'),
    ]
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='breath_works')
    breath_work_type = models.PositiveSmallIntegerField(choices=BREATH_WORK_CHOICES)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.user.identifier} - {self.get_breath_work_type_display()} at {self.created_at}"


# ============================================================================
# Blood Test / Lab Test Models
# ============================================================================

class BiomarkerDefinition(models.Model):
    """
    Defines all biomarkers that can be measured in lab tests.
    Stores reference ranges, optimal ranges, and metadata for each biomarker.
    Based on analysisEngine.ts _buildDynamicRanges()
    """
    
    CATEGORY_CHOICES = [
        ('hematology', 'هماتولوژی'),
        ('kidney', 'کلیه'),
        ('hormonal', 'هورمونی'),
        ('metabolic', 'متابولیک'),
        ('liver', 'کبد'),
        ('vitamins_minerals', 'ویتامین‌ها و مواد معدنی'),
        ('inflammation', 'التهاب'),
        ('thyroid', 'تیروئید'),
    ]
    
    # Basic Information
    name = models.CharField(max_length=100, unique=True, help_text="English name (e.g., 'Hb', 'ALT', 'HbA1c')")
    persian_name = models.CharField(max_length=200, blank=True, help_text="نام فارسی")
    full_name = models.CharField(max_length=200, blank=True, help_text="Full English name")
    category = models.CharField(max_length=50, choices=CATEGORY_CHOICES)
    unit = models.CharField(max_length=50, blank=True, help_text="Unit of measurement (e.g., 'mg/dL', 'U/L')")
    
    # Reference Ranges (Normal Range)
    norm_min_male = models.FloatField(null=True, blank=True, help_text="Normal minimum for males")
    norm_max_male = models.FloatField(null=True, blank=True, help_text="Normal maximum for males")
    norm_min_female = models.FloatField(null=True, blank=True, help_text="Normal minimum for females")
    norm_max_female = models.FloatField(null=True, blank=True, help_text="Normal maximum for females")
    norm_min_unisex = models.FloatField(null=True, blank=True, help_text="Normal minimum (gender-independent)")
    norm_max_unisex = models.FloatField(null=True, blank=True, help_text="Normal maximum (gender-independent)")
    
    # Optimal Ranges
    opt_min_male = models.FloatField(null=True, blank=True, help_text="Optimal minimum for males")
    opt_max_male = models.FloatField(null=True, blank=True, help_text="Optimal maximum for males")
    opt_min_female = models.FloatField(null=True, blank=True, help_text="Optimal minimum for females")
    opt_max_female = models.FloatField(null=True, blank=True, help_text="Optimal maximum for females")
    opt_min_unisex = models.FloatField(null=True, blank=True, help_text="Optimal minimum (gender-independent)")
    opt_max_unisex = models.FloatField(null=True, blank=True, help_text="Optimal maximum (gender-independent)")
    
    # Age-specific adjustments (stored as JSON for flexibility)
    age_adjustments = models.JSONField(default=dict, blank=True, help_text="Age-specific range adjustments")
    
    # Usage Information
    used_in_indices = models.JSONField(default=list, blank=True, help_text="Which indices use this biomarker (e.g., ['Cardiometabolic', 'Hematology'])")
    weight_in_indices = models.JSONField(default=dict, blank=True, help_text="Weights in each index (e.g., {'Cardiometabolic': 0.25})")
    used_in_risks = models.JSONField(default=list, blank=True, help_text="Which risk calculations use this (e.g., ['CVD', 'Diabetes'])")
    has_lifestyle_feedback = models.BooleanField(default=False, help_text="Has specific lifestyle feedback messages")
    
    # Metadata
    description = models.TextField(blank=True)
    is_active = models.BooleanField(default=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['category', 'name']
        verbose_name = 'Biomarker Definition'
        verbose_name_plural = 'Biomarker Definitions'
    
    def __str__(self):
        return f"{self.name} ({self.get_category_display()})"
    
    def get_norm_range(self, gender=None, age=None):
        """Get normal range based on gender and age."""
        if gender == 'male' or gender == 0:
            if self.norm_min_male is not None and self.norm_max_male is not None:
                return {'min': self.norm_min_male, 'max': self.norm_max_male}
        elif gender == 'female' or gender == 1:
            if self.norm_min_female is not None and self.norm_max_female is not None:
                return {'min': self.norm_min_female, 'max': self.norm_max_female}
        
        if self.norm_min_unisex is not None and self.norm_max_unisex is not None:
            return {'min': self.norm_min_unisex, 'max': self.norm_max_unisex}
        
        return None
    
    def get_opt_range(self, gender=None, age=None):
        """Get optimal range based on gender and age."""
        if gender == 'male' or gender == 0:
            if self.opt_min_male is not None and self.opt_max_male is not None:
                return {'min': self.opt_min_male, 'max': self.opt_max_male}
        elif gender == 'female' or gender == 1:
            if self.opt_min_female is not None and self.opt_max_female is not None:
                return {'min': self.opt_min_female, 'max': self.opt_max_female}
        
        if self.opt_min_unisex is not None and self.opt_max_unisex is not None:
            return {'min': self.opt_min_unisex, 'max': self.opt_max_unisex}
        
        return None


class LabTest(models.Model):
    """
    Represents a complete blood test / lab test report for a patient.
    LabTest is a subset of User profile - patient information comes from User.profile.
    This model only stores test-specific information, not duplicate patient data.
    """
    
    # REQUIRED: LabTest must belong to a User (patient profile)
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='lab_tests', help_text="بیمار - اطلاعات از پروفایل کاربر گرفته می‌شود")
    
    # Test Information
    test_date = models.DateField(null=True, blank=True, help_text="تاریخ آزمایش")
    collection_date = models.DateField(null=True, blank=True, help_text="تاریخ نمونه‌گیری")
    confirm_date = models.DateField(null=True, blank=True, help_text="تاریخ تایید")
    lab_center = models.CharField(max_length=200, blank=True, help_text="مرکز آزمایشگاه")
    registration_number = models.CharField(max_length=100, blank=True, help_text="کد پذیرش")
    
    # Source Information
    source_file = models.FileField(upload_to='lab_tests/source_files/', null=True, blank=True, help_text="فایل Excel/PDF اصلی")
    source_type = models.CharField(max_length=20, choices=[('excel', 'Excel'), ('pdf', 'PDF'), ('manual', 'Manual')], default='manual')
    
    # Analysis Results (stored as JSON for flexibility)
    overall_health_score = models.IntegerField(null=True, blank=True, help_text="امتیاز کلی سلامت (0-100)")
    indices = models.JSONField(default=dict, blank=True, help_text="Health indices scores")
    index_details = models.JSONField(default=dict, blank=True, help_text="Detailed index analysis with test results")
    risks = models.JSONField(default=dict, blank=True, help_text="Risk calculations (CVD, Diabetes, Liver)")
    raw_data = models.JSONField(default=dict, blank=True, help_text="Raw biomarker values from source")
    
    # Metadata
    is_analyzed = models.BooleanField(default=False, help_text="Whether analysis has been performed")
    analyzed_at = models.DateTimeField(null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    created_by = models.ForeignKey(DjangoUser, on_delete=models.SET_NULL, null=True, blank=True, help_text="Doctor/User who uploaded this")
    
    class Meta:
        ordering = ['-test_date', '-created_at']
        verbose_name = 'Lab Test'
        verbose_name_plural = 'Lab Tests'
        indexes = [
            models.Index(fields=['user', '-test_date']),
        ]
    
    def __str__(self):
        patient_name = self.get_patient_name()
        return f"Lab Test - {patient_name} - {self.test_date or 'No Date'}"
    
    def get_patient_name(self):
        """Get patient name from user profile."""
        if self.user and self.user.profile:
            return f"{self.user.profile.first_name} {self.user.profile.last_name}".strip()
        return "Unknown"
    
    def get_patient_national_id(self):
        """Get national ID from user profile."""
        if self.user and self.user.profile:
            return self.user.profile.national_id
        return None
    
    def get_patient_gender(self):
        """Get gender from user profile."""
        if self.user and self.user.profile:
            return 'female' if self.user.profile.gender else 'male'
        return 'unknown'
    
    def get_patient_age(self):
        """Calculate age from user profile birth_date."""
        if self.user and self.user.profile and self.user.profile.birth_date:
            from datetime import date
            today = date.today()
            birth_date = self.user.profile.birth_date
            if hasattr(birth_date, 'date'):
                birth_date = birth_date.date()
            elif isinstance(birth_date, datetime):
                birth_date = birth_date.date()
            return today.year - birth_date.year - ((today.month, today.day) < (birth_date.month, birth_date.day))
        return None
    
    @staticmethod
    def calculate_age_from_birth_date(birth_date):
        """Calculate age from birth_date."""
        if not birth_date:
            return None
        from datetime import date
        today = date.today()
        if hasattr(birth_date, 'date'):
            birth_date = birth_date.date()
        elif isinstance(birth_date, datetime):
            birth_date = birth_date.date()
        return today.year - birth_date.year - ((today.month, today.day) < (birth_date.month, birth_date.day))
    
    @classmethod
    def find_or_create_user_from_lab_data(cls, first_name, last_name, national_id=None, gender=None, age=None, created_by=None):
        """
        Find existing user by name/national_id or create new user profile.
        Updates user profile with lab test data if missing.
        If new user is created, assigns them to the same company groups as the uploader.
        
        Args:
            first_name: First name from lab test
            last_name: Last name from lab test
            national_id: National ID from lab test
            gender: Gender from lab test
            age: Age from lab test
            created_by: DjangoUser who uploaded the lab test (used to determine company assignment)
        
        Returns: (user, created)
        """
        from datetime import datetime
        from django.utils import timezone
        
        user = None
        created = False
        
        # Try to find by national_id first (most reliable)
        if national_id and national_id.strip():
            try:
                user = User.objects.filter(profile__national_id=national_id).first()
            except Exception:
                pass
        
        # If not found, try to find by name
        if not user and first_name and last_name:
            first_name_clean = first_name.strip()
            last_name_clean = last_name.strip()
            try:
                # Try exact match
                user = User.objects.filter(
                    profile__first_name__iexact=first_name_clean,
                    profile__last_name__iexact=last_name_clean
                ).first()
            except Exception:
                pass
        
        # Create new user if not found
        if not user:
            # Create Django User
            username = f"lab_{national_id or f'{first_name}_{last_name}'}_{int(timezone.now().timestamp())}"
            django_user = DjangoUser.objects.create_user(
                username=username,
                password=None,  # Will be set later if needed
            )
            
            # Create User
            user = User.objects.create(django_user=django_user)
            
            # Create Profile for the new user
            profile = Profile.objects.create()
            user.profile = profile
            user.save()
            created = True
        
        # Get or create profile (handle case where existing user might not have profile)
        profile = user.profile
        if not profile:
            profile = Profile.objects.create()
            user.profile = profile
            user.save()
        
        # Update profile with lab test data (only if missing or different)
        # This ensures first_name, last_name, and age are read from the lab test
        updated = False
        
        if national_id and national_id.strip() and (not profile.national_id or profile.national_id != national_id):
            profile.national_id = national_id
            updated = True
        
        if first_name and first_name.strip() and (not profile.first_name or profile.first_name != first_name.strip()):
            profile.first_name = first_name.strip()
            updated = True
        
        if last_name and last_name.strip() and (not profile.last_name or profile.last_name != last_name.strip()):
            profile.last_name = last_name.strip()
            updated = True
        
        if gender and gender != 'unknown':
            # Convert 'male'/'female' to boolean (0=male, 1=female)
            gender_bool = 1 if gender.lower() == 'female' else 0
            if profile.gender != gender_bool:
                profile.gender = gender_bool
                updated = True
        
        if age and age > 0:
            # Calculate birth_date from age (approximate)
            current_age = cls.calculate_age_from_birth_date(profile.birth_date) if profile.birth_date else None
            if not profile.birth_date or (current_age and abs(current_age - age) > 1):
                # Set birth_date to approximate age
                from datetime import date
                birth_year = date.today().year - age
                profile.birth_date = datetime(birth_year, 1, 1)
                updated = True
        
        if updated or created:
            profile.save()
        
        return user, created


class LabTestResult(models.Model):
    """
    Individual biomarker result within a lab test.
    Links a specific biomarker to its measured value in a lab test.
    """
    
    lab_test = models.ForeignKey(LabTest, on_delete=models.CASCADE, related_name='results')
    biomarker = models.ForeignKey(BiomarkerDefinition, on_delete=models.CASCADE, related_name='results', null=True, blank=True)
    biomarker_name = models.CharField(max_length=100, help_text="Name of biomarker (if not linked to definition)")
    
    # Test Result
    value = models.FloatField(null=True, blank=True, help_text="Numeric value")
    value_string = models.CharField(max_length=200, blank=True, help_text="Original string value (e.g., 'Negative', '<5')")
    unit = models.CharField(max_length=50, blank=True, help_text="Unit of measurement")
    method = models.CharField(max_length=100, blank=True, help_text="Test method")
    
    # Reference Range (from lab report)
    ref_min = models.FloatField(null=True, blank=True, help_text="Reference minimum from lab")
    ref_max = models.FloatField(null=True, blank=True, help_text="Reference maximum from lab")
    ref_label = models.CharField(max_length=200, blank=True, help_text="Reference range label")
    
    # Analysis Results
    fuzzy_score = models.IntegerField(null=True, blank=True, help_text="Calculated fuzzy score (0-100)")
    status = models.CharField(max_length=50, choices=[
        ('یافت نشد', 'یافت نشد'),
        ('ایده‌آل', 'ایده‌آل'),
        ('طبیعی', 'طبیعی'),
        ('نیازمند بررسی', 'نیازمند بررسی'),
    ], blank=True)
    lifestyle_feedback = models.TextField(blank=True, help_text="توصیه‌های سبک زندگی")
    
    # Metadata
    test_code = models.CharField(max_length=50, blank=True, help_text="کد آزمایش")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['lab_test', 'biomarker_name']
        verbose_name = 'Lab Test Result'
        verbose_name_plural = 'Lab Test Results'
        unique_together = [['lab_test', 'biomarker_name']]
        indexes = [
            models.Index(fields=['lab_test', 'biomarker']),
        ]
    
    def __str__(self):
        biomarker_display = self.biomarker.name if self.biomarker else self.biomarker_name
        value_display = self.value if self.value is not None else self.value_string
        patient_name = self.lab_test.get_patient_name() if self.lab_test else "Unknown"
        return f"{biomarker_display}: {value_display} ({patient_name})"
    
    @property
    def display_name(self):
        """Get display name for biomarker."""
        if self.biomarker:
            return self.biomarker.persian_name or self.biomarker.name
        return self.biomarker_name
    
    @property
    def display_value(self):
        """Get display value."""
        if self.value is not None:
            return str(self.value)
        return self.value_string or "-"


class AIChatMessage(models.Model):
    """
    Model to store AI chat conversations between users and the AI assistant.
    Stores user messages and AI responses for the Lacto AI Coach feature.
    """
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='ai_chat_messages')
    user_message = models.TextField(help_text="The message sent by the user")
    ai_response = models.TextField(help_text="The response generated by the AI")
    created_at = models.DateTimeField(default=datetime.now, help_text="Timestamp when the message was created")
    conversation_id = models.CharField(max_length=100, blank=True, null=True, 
                                      help_text="Optional conversation ID to group related messages")
    
    class Meta:
        ordering = ['-created_at']
        verbose_name = 'AI Chat Message'
        verbose_name_plural = 'AI Chat Messages'
        indexes = [
            models.Index(fields=['user', '-created_at']),
            models.Index(fields=['conversation_id']),
        ]
    
    def __str__(self):
        user_name = self.user.profile.first_name if self.user.profile else str(self.user.id)
        message_preview = self.user_message[:50] + "..." if len(self.user_message) > 50 else self.user_message
        return f"AI Chat - {user_name}: {message_preview}"


class FoodRecommendationsCache(models.Model):
    """
    Model to store food recommendations for users.
    Stores all recommendations in one JSON record for extremely fast retrieval.
    """
    user = models.OneToOneField(User, on_delete=models.CASCADE, related_name='food_recommendations_cache')
    recommendations = models.JSONField(default=list, help_text="Complete list of food recommendations")
    diseases_string = models.TextField(blank=True, null=True, help_text="Diseases string used to generate recommendations")
    lab_results = models.JSONField(default=list, blank=True, null=True, help_text="Lab results used to generate recommendations")
    total_foods = models.IntegerField(default=0, help_text="Total number of food recommendations")
    created_at = models.DateTimeField(default=datetime.now, help_text="When recommendations were generated")
    updated_at = models.DateTimeField(auto_now=True, help_text="Last update timestamp")
    
    class Meta:
        verbose_name = 'Food Recommendations Cache'
        verbose_name_plural = 'Food Recommendations Caches'
        indexes = [
            models.Index(fields=['user']),
            models.Index(fields=['-updated_at']),
        ]
    
    def __str__(self):
        user_name = self.user.profile.first_name if self.user.profile else str(self.user.id)
        return f"Food Recommendations Cache - {user_name} ({self.total_foods} foods)"