from django.contrib import admin
from .forms import FoodCardForm
from django.urls import resolvers
from django.utils.html import format_html
from .models import *
from panel.models import *
from django.db.models import Avg, Max, Count, Q
from .models import Mood


class FoodAdmin(admin.ModelAdmin):
    list_display = [
        field.name for field in Food._meta.get_fields()
        if not field.many_to_many and not field.one_to_many
    ]
    search_fields = list_display


admin.site.register(Food, FoodAdmin)

class MicrobiomeQuestionnaireAdmin(admin.ModelAdmin):
    list_display = []
    list_display.append("user")
    search_fields = list_display
admin.site.register(MicrobiomeQuestionnaire, MicrobiomeQuestionnaireAdmin)  

class LifestyleQuestionnaireAdmin(admin.ModelAdmin):
    list_display = []
    list_display.append("user")
    search_fields = list_display    
admin.site.register(LifestyleQuestionnaire, LifestyleQuestionnaireAdmin)

class UserAdmin(admin.ModelAdmin):
    list_display = (
        'identifier',
        'phone_number',
        'id',
        'profile_id',
        'token',
        'django_user_id',
        'isRegistered',
        'isLogedin',
        'created_at'
    )
    search_fields = ['phone_number', 'token', 'id']
    list_filter = ('isRegistered', 'isLogedin', 'created_at')  # Added filters


admin.site.register(User, UserAdmin)


class TempTokenAdmin(admin.ModelAdmin):
    list_display = ('phone_number', "activation_code", "temp_token", "time")
    search_fields = ['phone_number']


admin.site.register(TempToken, TempTokenAdmin)


class ProfileAdmin(admin.ModelAdmin):
    list_display = ('id',
                    'first_name',
                    'last_name',
                    'hospital_code',
                    'register_time',
                    'gender',
                    'age',
                    'birth_date',
                    'user_code',
                    'sub_users',
                    'intro_code',
                    'acquisition_source',
                    'acquisition_date',
                    'weight',
                    'goal_weight',
                    'height',
                    'user_setting',
                    'activity_level',
                    'CR',
                    'BMR',
                    'Carbohydrates_g',
                    'Protein_g',
                    'Fat_g',
                    'Carbohydrates_unit',
                    'Protein_unit',
                    'Fat_unit',
                    'Carbohydrates_distribution_list_g',
                    'Protein_distribution_g',
                    'Fat_distribution_g',
                    'a1c',
                    'device_type',
                    'email',
                    "firebase_token"
                    )
    search_fields = ['^first_name', '^last_name', '=id', 'email', 'user__phone_number', 'acquisition_details']
    list_filter = ('gender', 'register_time', 'acquisition_source')


admin.site.register(Profile, ProfileAdmin)


class InsulinAdmin(admin.ModelAdmin):
    list_display = ('name',
                    'Data_Base_Number',
                    'EffectType',
                    'Company',
                    'id',
                    'coefficient')
    search_fields = ['name']


admin.site.register(Insulin, InsulinAdmin)


class ReportAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'created_at',
                    'start_time',
                    'end_time',
                    'message',
                    'code')
    search_fields = ['user__phone_number']


admin.site.register(Report, ReportAdmin)


class ActivityAdmin(admin.ModelAdmin):
    list_display = ('name',
                    'Metz',)
    search_fields = ['name']


admin.site.register(Activity, ActivityAdmin)


class DrugAdmin(admin.ModelAdmin):
    list_display = ('name',
                    'EN_Name',
                    'description',
                    'id',
                    'DataBaseNumber')
    search_fields = ['name']


admin.site.register(Drug, DrugAdmin)


class GLP1Admin(admin.ModelAdmin):
    list_display = ('name',
                    'EN_Name',
                    'description',
                    'id',
                    'DataBaseNumber')
    search_fields = ['name']


admin.site.register(GLP1, GLP1Admin)


class ActivityLogsAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'activity',
                    'amount',
                    'start_date_time',
                    'point',
                    'PALx')
    search_fields = ['user']


admin.site.register(Activities_log, ActivityLogsAdmin)


class SleepLogsAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'start',
                    'end',
                    'time',
                    'sleeped')
    search_fields = ['user']


admin.site.register(Sleep, SleepLogsAdmin)


class EatingAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'food',
                    'date_time',
                    'amount',
                    'meal',
                    'point')
    search_fields = ['user__phone_number']


admin.site.register(Eating, EatingAdmin)


class SugarMeasurementAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'amount',
                    'meal',
                    'time',
                    'hypo',
                    'hyper',
                    'point')
    search_fields = ['user']


admin.site.register(SugarMeasurement, SugarMeasurementAdmin)


class FeelingAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'amount',
                    'name',
                    'time',
                    'description',
                    'point')
    search_fields = ['user']


admin.site.register(Feeling, FeelingAdmin)


class InsulinUsageAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'amount',
                    'meal',
                    'Insulin',
                    'time',
                    'point')
    search_fields = ['user']


admin.site.register(InsulinUsage, InsulinUsageAdmin)


class DrugUsageAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'amount',
                    'meal',
                    'drug',
                    'time',
                    'point')
    search_fields = ['user']


admin.site.register(DrugUsage, DrugUsageAdmin)


class WeightsAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'weight',
                    'time',
                    'BMI')
    search_fields = ['user']


admin.site.register(Weights, WeightsAdmin)


class PointsAdmin(admin.ModelAdmin):
    list_display = []
    for field in Points._meta.get_fields():
        list_display.append(field.name)
    search_fields = list_display


admin.site.register(Points, PointsAdmin)


class WalkingAdmin(admin.ModelAdmin):
    list_display = ('user',
                    'steps',
                    'amount',
                    'date',
                    'PALx')
    search_fields = ['user']


admin.site.register(Walking, WalkingAdmin)


class NotificationAdmin(admin.ModelAdmin):
    list_display = ('Title',
                    'Link',
                    'datetime',
                    'Summary',
                    'id')
    search_fields = ['title']


admin.site.register(Notification, NotificationAdmin)


class UpdateTableAdmin(admin.ModelAdmin):
    list_display = ('DateTime',
                    'Type',
                    'FileUpdateName')
    search_fields = ['Type']


admin.site.register(UpdateTable, UpdateTableAdmin)


class APICALLAdmin(admin.ModelAdmin):
    list_display = ('date',
                    'error_transactions',
                    'all_transactions')
    search_fields = ['date']


admin.site.register(Transaction, APICALLAdmin)


class GiftAdmin(admin.ModelAdmin):
    list_display = ('id',
                    'created_at',
                    'name',
                    'user',
                    'value',
                    'code',
                    'link',
                    'description')
    search_fields = ['name']


admin.site.register(Gift, GiftAdmin)


class DietAdmin(admin.ModelAdmin):
    list_display = []
    for field in Diet._meta.get_fields():
        list_display.append(field.name)
    list_display.remove('diet_json')
    list_display.remove('eaten_foods')
    search_fields = ['user', 'creator']


admin.site.register(Diet, DietAdmin)


class DietAdminTemplate(admin.ModelAdmin):
    list_display = []
    for field in DietTemplate._meta.get_fields():
        list_display.append(field.name)
    list_display.remove('diet_json')
    search_fields = ['diet_name', 'creator']


admin.site.register(DietTemplate, DietAdminTemplate)


class AdminAdmin(admin.ModelAdmin):
    list_display = []
    for field in Admins._meta.get_fields():
        list_display.append(field.name)
    search_fields = list_display


admin.site.register(Admins, AdminAdmin)

# class AddressAdmin(admin.ModelAdmin):
#     list_display = []
#     for field in Address._meta.get_fields():
#         list_display.append(field.name)
#     search_fields = list_display
#
# admin.site.register(Address, AddressAdmin)

admin.site.register(Address)


class UserQuestionnaireAdmin(admin.ModelAdmin):
    list_display = []
    for field in UserQuestionnaire._meta.get_fields():
        list_display.append(field.name)
    search_fields = list_display


admin.site.register(UserQuestionnaire, UserQuestionnaireAdmin)


class ZiLuckGroupAdmin(admin.ModelAdmin):
    list_display = ('id', 'name', 'group_id', 'is_company', 'is_public', 'get_members_count', 'admin', 'creator', 'created_at')
    list_filter = ('is_company', 'is_public', 'created_at', 'admin', 'creator')
    search_fields = ('name', 'group_id', 'id')
    readonly_fields = ('created_at', 'get_members_count', 'link')
    date_hierarchy = 'created_at'
    ordering = ('-created_at',)
    
    fieldsets = (
        ('Basic Information', {
            'fields': ('name', 'group_id', 'is_company', 'is_public')
        }),
        ('Group Management', {
            'fields': ('admin', 'creator', 'get_members_count', 'link')
        }),
        ('System Information', {
            'fields': ('created_at',),
            'classes': ('collapse',)
        }),
    )
    
    def get_members_count(self, obj):
        """Display number of members in the group"""
        return obj.members_number
    get_members_count.short_description = 'Members Count'
    get_members_count.admin_order_field = 'user_set__count'
    
    def get_queryset(self, request):
        """Optimize queryset with annotations"""
        qs = super().get_queryset(request)
        return qs.select_related('admin', 'creator').prefetch_related('user_set')
    
    def changelist_view(self, request, extra_context=None):
        """Add company filter to the changelist"""
        extra_context = extra_context or {}
        # Add custom filter for companies
        if request.GET.get('is_company__exact') == '1':
            extra_context['title'] = 'Company Groups'
        elif request.GET.get('is_company__exact') == '0':
            extra_context['title'] = 'Regular Groups'
        return super().changelist_view(request, extra_context)


admin.site.register(ZiLuckGroup, ZiLuckGroupAdmin)


class FoodCardAdmin(admin.ModelAdmin):
    list_display = []
    for field in FoodCard._meta.get_fields():
        list_display.append(field.name)
    # Only search on text fields and exclude foreign keys
    search_fields = ['FA_Name', 'EN_Name', 'recipe_description', 'recipe_image_link']
    list_filter = ['is_keto', 'is_med', 'is_conditional_med', 'is_lowcarb', 'is_mediumcarb', 'is_highcarb', 
                  'is_highprotein', 'is_lowprotein', 'is_mediumprotein', 'is_mind', 'is_lowfat', 
                  'is_mediumfat', 'is_highfat', 'is_low_fodmap', 'is_breakfast', 'is_lunch', 
                  'is_dinner', 'is_snack']


admin.site.register(FoodCard, FoodCardAdmin)


class MicrobiomeAnalysisAdmin(admin.ModelAdmin):
    list_display = []
    for field in MicrobiomeAnalysis._meta.get_fields():
        list_display.append(field.name)
    search_fields = list_display


admin.site.register(MicrobiomeAnalysis, MicrobiomeAnalysisAdmin)


class PurchaseAdmin(admin.ModelAdmin):
    # Define fields to display explicitly, excluding reverse relations if any
    list_display = ['id', 'user', 'shop_item', 'purchase_date']
    search_fields = ['user__username', 'shop_item__name']  # Customize search fields as needed


admin.site.register(Purchase, PurchaseAdmin)


class ShopAdmin(admin.ModelAdmin):
    # Define fields to display explicitly, excluding problematic relationships
    list_display = ['id', 'name', 'price', 'is_coin', 'description', 'tag', 'service_type', 'promotion_percentage',
                    'promotion_end_date']
    search_fields = ['name', 'tag', 'price']  # Customize search fields as needed


admin.site.register(Shop, ShopAdmin)


class DietSubscriptionAdmin(admin.ModelAdmin):
    # Define fields to display explicitly, excluding problematic relationships
    list_display = ['id', 'user', 'diet_plan', 'start_date', 'duration_weeks', 'end_date']
    search_fields = ['user', 'duration_weeks', 'diet_plan']  # Customize search fields as needed


admin.site.register(DietSubscription, DietSubscriptionAdmin)


class AppointmentCreditAdmin(admin.ModelAdmin):
    # Define fields to display explicitly, excluding problematic relationships
    list_display = ['id', 'user', 'purchase', 'total_credits', 'used_credits']
    search_fields = ['user', 'total_credits', 'purchase']  # Customize search fields as needed


admin.site.register(AppointmentCredit, AppointmentCreditAdmin)


class WaterAdmin(admin.ModelAdmin):
    list_display = []
    for field in Water._meta.get_fields():
        list_display.append(field.name)
    search_fields = list_display


admin.site.register(Water, WaterAdmin)

from django.utils.safestring import mark_safe


class AnalysisAdmin(admin.ModelAdmin):
    list_display = []
    readonly_fields = ('display_api_monitoring',)
    for field in Analysis._meta.get_fields():
        list_display.append(field.name)
    list_display.insert(1, 'display_api_monitoring')
    search_fields = list_display
    search_fields.remove("api_monitoring")

    def display_api_monitoring(self, obj):
        if not obj.api_monitoring:
            return "No data"

        html = "<table style='border:1px solid black;'>"
        data = obj.api_monitoring.get('report', [])
        for item in data:
            url = item[0]
            details = item[1]
            true_count = details.get("true", 0)
            false_count = details.get("false", 0)
            errors = details.get("errors", [])

            html += f"<tr><td style='border:1px solid black;'>{url}</td></tr>"
            html += "<tr><td><table style='border:1px solid black;'>"
            html += f"<tr><td style='border:1px solid black;'>True</td><td style='border:1px solid black;'>{true_count}</td></tr>"
            html += f"<tr><td style='border:1px solid black;'>False</td><td style='border:1px solid black;'>{false_count}</td></tr>"
            html += "<tr><td style='border:1px solid black;'>Errors</td><td><ul>"
            for error in errors:
                html += f"<li>{error}</li>"
            html += "</ul></td></tr>"
            html += "</table></td></tr>"

        html += "</table>"
        return mark_safe(html)

    display_api_monitoring.short_description = 'API Monitoring'


admin.site.register(Analysis, AnalysisAdmin)


class UserScriptAdmin(admin.ModelAdmin):
    list_display = []
    for field in UserScript._meta.get_fields():
        list_display.append(field.name)
    search_fields = list_display


admin.site.register(UserScript, UserScriptAdmin)


@admin.register(ApiRequestLog)
class ApiRequestLogAdmin(admin.ModelAdmin):
    list_display = ('url', 'response_time', 'timestamp')
    list_filter = ('url', 'timestamp')  # Filters for the main list
    search_fields = ('url',)
    date_hierarchy = 'timestamp'

    def changelist_view(self, request, extra_context=None):
        # Retrieve the filter option from the GET parameter in the request
        filter_option = request.GET.get('date_filter', 'all')

        # Calculate the start date based on the filter option
        start_date = None
        if filter_option == 'today':
            start_date = datetime.now().date()
        elif filter_option == 'week':
            start_date = datetime.now() - timedelta(days=7)
        elif filter_option == 'month':
            start_date = datetime.now() - timedelta(days=30)

        # Define a filter that can be used for both tables
        queryset_filter = Q() if start_date is None else Q(timestamp__gte=start_date)

        # Filter the display list (left table)
        self.list_display_links = None  # Remove unnecessary links
        self.queryset = lambda request: ApiRequestLog.objects.filter(queryset_filter)

        # Calculate URL statistics (right table)
        url_stats = self.get_url_stats(start_date)

        # Pass the statistics to the template
        extra_context = extra_context or {}
        extra_context['url_stats'] = url_stats
        extra_context['filter_option'] = filter_option  # To display the current option on the page

        return super().changelist_view(request, extra_context=extra_context)

    @staticmethod
    def get_url_stats(start_date=None):
        """
        Calculate statistics for URLs with optional date filtering.
        """
        queryset = ApiRequestLog.objects.all()

        # Apply the filter based on the start date
        if start_date:
            queryset = queryset.filter(timestamp__gte=start_date)

        # Calculate statistics
        stats = list(
            queryset.values('url').annotate(
                avg=Avg('response_time'),
                max=Max('response_time'),
                count=Count('id')
            ).values('url', 'avg', 'max', 'count')
        )

        # Sort by average response time
        stats.sort(key=lambda x: x['avg'], reverse=True)

        # Return statistics as a dictionary
        return {item['url']: {'avg': item['avg'], 'max': item['max'], 'count': item['count']} for item in stats}



class FreeDietPlanAdmin(admin.ModelAdmin):
    list_display = []
    for field in FreeDietPlan._meta.get_fields():
        list_display.append(field.name)
    search_fields = list_display


admin.site.register(FreeDietPlan, FreeDietPlanAdmin)


@admin.register(NotificationLog)
class NotificationLogAdmin(admin.ModelAdmin):
    list_display = ('user', 'title', 'sent_at', 'status', 'error_message')
    list_filter = ('status', 'sent_at')
    search_fields = ('user__phone_number', 'title', 'body')
    date_hierarchy = 'sent_at'


class WellnessProfileAdmin(admin.ModelAdmin):
    list_display = []
    list_display.append('id')
    list_display.append('user')
    search_fields = list_display

admin.site.register(WellnessProfile, WellnessProfileAdmin)

# Resource Hub Admin Models
class CategoryAdmin(admin.ModelAdmin):
    list_display = ('name', 'slug', 'parent', 'creator', 'is_active', 'ordering', 'resource_count', 'created_at')
    list_filter = ('is_active', 'creator', 'created_at')
    search_fields = ('name', 'description', 'slug')
    prepopulated_fields = {'slug': ('name',)}
    ordering = ('parent__name', 'ordering', 'name')

admin.site.register(Category, CategoryAdmin)

class ResourceAdmin(admin.ModelAdmin):
    list_display = ('title', 'slug', 'category', 'type', 'creator', 'is_published', 'is_featured', 'view_count', 'ordering', 'created_at')
    list_filter = ('type', 'is_published', 'is_featured', 'category', 'creator', 'created_at')
    search_fields = ('title', 'description', 'slug')
    prepopulated_fields = {'slug': ('title',)}
    ordering = ('category__name', 'ordering', 'title')
    readonly_fields = ('view_count',)

admin.site.register(Resource, ResourceAdmin)

class VideoResourceAdmin(admin.ModelAdmin):
    list_display = ('resource', 'video_url', 'duration_minutes', 'thumbnail')
    search_fields = ('resource__title', 'video_url')
    list_filter = ('duration_minutes',)

admin.site.register(VideoResource, VideoResourceAdmin)

class PDFResourceAdmin(admin.ModelAdmin):
    list_display = ('resource', 'file', 'file_size', 'page_count')
    search_fields = ('resource__title',)
    list_filter = ('page_count',)

admin.site.register(PDFResource, PDFResourceAdmin)

class ImageSeriesResourceAdmin(admin.ModelAdmin):
    list_display = ('resource', 'items_count')
    search_fields = ('resource__title',)

    def items_count(self, obj):
        return obj.items.count()
    items_count.short_description = 'Items Count'

admin.site.register(ImageSeriesResource, ImageSeriesResourceAdmin)

class ImageSeriesItemAdmin(admin.ModelAdmin):
    list_display = ('image_series', 'caption', 'ordering', 'image')
    list_filter = ('ordering',)
    search_fields = ('caption', 'image_series__resource__title')
    ordering = ('image_series', 'ordering')

admin.site.register(ImageSeriesItem, ImageSeriesItemAdmin)

class UserResourceInteractionAdmin(admin.ModelAdmin):
    list_display = ('user', 'resource', 'viewed', 'completed', 'bookmarked', 'liked', 'progress', 'time_spent_seconds', 'last_viewed_at')
    list_filter = ('viewed', 'completed', 'bookmarked', 'liked', 'resource__type', 'resource__category')
    search_fields = ('user__username', 'resource__title')
    readonly_fields = ('first_viewed_at', 'last_viewed_at', 'completed_at')
    ordering = ('-last_viewed_at',)

admin.site.register(UserResourceInteraction, UserResourceInteractionAdmin)

class UserCategoryProgressAdmin(admin.ModelAdmin):
    list_display = ('user', 'category', 'total_resources', 'viewed_resources', 'completed_resources', 'completion_percent', 'updated_at')
    list_filter = ('category', 'updated_at')
    search_fields = ('user__username', 'category__name')
    readonly_fields = ('updated_at',)
    ordering = ('-updated_at',)

admin.site.register(UserCategoryProgress, UserCategoryProgressAdmin)

admin.site.register(Mood)

class CommentAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'user_identifier',
        'content_preview',
        'is_active',
        'created_at',
        'updated_at'
    )
    list_filter = (
        'is_active',
        'created_at',
        'updated_at'
    )
    search_fields = (
        'content',
        'user__phone_number',
        'user__profile__first_name',
        'user__profile__last_name'
    )
    readonly_fields = ('created_at', 'updated_at')
    ordering = ('-created_at',)
    list_per_page = 25
    
    def user_identifier(self, obj):
        """Display user identifier in list view"""
        return obj.user.identifier if obj.user else "Unknown User"
    user_identifier.short_description = 'User'
    user_identifier.admin_order_field = 'user__phone_number'
    
    def content_preview(self, obj):
        """Display truncated content preview"""
        if len(obj.content) > 50:
            return obj.content[:50] + "..."
        return obj.content
    content_preview.short_description = 'Content Preview'
    
    fieldsets = (
        ('Comment Information', {
            'fields': ('user', 'content', 'is_active')
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )

admin.site.register(Comment, CommentAdmin)


# Lab Test Models Admin
class BiomarkerDefinitionAdmin(admin.ModelAdmin):
    list_display = ('name', 'persian_name', 'full_name', 'category', 'unit', 'is_active', 'created_at')
    list_filter = ('category', 'is_active', 'created_at')
    search_fields = ('name', 'persian_name', 'full_name', 'description')
    ordering = ('category', 'name')
    readonly_fields = ('created_at', 'updated_at')
    
    fieldsets = (
        ('Basic Information', {
            'fields': ('name', 'persian_name', 'full_name', 'category', 'unit', 'description', 'is_active')
        }),
        ('Reference Ranges - Male', {
            'fields': ('norm_min_male', 'norm_max_male', 'opt_min_male', 'opt_max_male'),
            'classes': ('collapse',)
        }),
        ('Reference Ranges - Female', {
            'fields': ('norm_min_female', 'norm_max_female', 'opt_min_female', 'opt_max_female'),
            'classes': ('collapse',)
        }),
        ('Reference Ranges - Unisex', {
            'fields': ('norm_min_unisex', 'norm_max_unisex', 'opt_min_unisex', 'opt_max_unisex'),
            'classes': ('collapse',)
        }),
        ('Usage Information', {
            'fields': ('age_adjustments', 'used_in_indices', 'weight_in_indices', 'used_in_risks', 'has_lifestyle_feedback'),
            'classes': ('collapse',)
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )


admin.site.register(BiomarkerDefinition, BiomarkerDefinitionAdmin)


class LabTestResultInline(admin.TabularInline):
    """Inline admin for LabTestResult within LabTest"""
    model = LabTestResult
    extra = 0
    fields = ('biomarker', 'biomarker_name', 'value', 'value_string', 'unit', 'fuzzy_score', 'status')
    readonly_fields = ('fuzzy_score', 'status', 'lifestyle_feedback')
    show_change_link = True


class LabTestAdmin(admin.ModelAdmin):
    list_display = ('id', 'get_patient_name', 'get_patient_national_id', 'test_date', 'lab_center', 'overall_health_score', 'is_analyzed', 'created_at')
    list_filter = ('is_analyzed', 'source_type', 'test_date', 'created_at', 'lab_center')
    search_fields = ('user__profile__first_name', 'user__profile__last_name', 'user__profile__national_id', 'lab_center', 'registration_number')
    readonly_fields = ('created_at', 'updated_at', 'analyzed_at', 'get_patient_name', 'get_patient_national_id', 'get_patient_gender', 'get_patient_age')
    date_hierarchy = 'test_date'
    inlines = [LabTestResultInline]
    
    fieldsets = (
        ('Patient Information', {
            'fields': ('user', 'get_patient_name', 'get_patient_national_id', 'get_patient_gender', 'get_patient_age'),
            'description': 'Patient information comes from User profile (no duplicates stored here)'
        }),
        ('Test Information', {
            'fields': ('test_date', 'collection_date', 'confirm_date', 'lab_center', 'registration_number')
        }),
        ('Source Information', {
            'fields': ('source_file', 'source_type', 'created_by')
        }),
        ('Analysis Results', {
            'fields': ('overall_health_score', 'indices', 'index_details', 'risks', 'raw_data', 'is_analyzed', 'analyzed_at'),
            'classes': ('collapse',)
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    def get_patient_name(self, obj):
        """Display patient name from profile"""
        return obj.get_patient_name()
    get_patient_name.short_description = 'Patient Name'
    
    def get_patient_national_id(self, obj):
        """Display national ID from profile"""
        return obj.get_patient_national_id()
    get_patient_national_id.short_description = 'National ID'
    
    def get_patient_gender(self, obj):
        """Display gender from profile"""
        gender = obj.get_patient_gender()
        return 'مرد' if gender == 'male' else ('زن' if gender == 'female' else 'نامشخص')
    get_patient_gender.short_description = 'Gender'
    
    def get_patient_age(self, obj):
        """Display age from profile"""
        age = obj.get_patient_age()
        return f"{age} years" if age else "Unknown"
    get_patient_age.short_description = 'Age'


admin.site.register(LabTest, LabTestAdmin)


class LabTestResultAdmin(admin.ModelAdmin):
    list_display = ('id', 'lab_test', 'biomarker', 'biomarker_name', 'value', 'value_string', 'unit', 'fuzzy_score', 'status', 'created_at')
    list_filter = ('status', 'biomarker__category', 'lab_test__test_date', 'created_at')
    search_fields = ('biomarker_name', 'biomarker__name', 'biomarker__persian_name', 'lab_test__user__profile__first_name', 'lab_test__user__profile__last_name')
    readonly_fields = ('fuzzy_score', 'status', 'lifestyle_feedback', 'created_at', 'updated_at')
    raw_id_fields = ('lab_test', 'biomarker')
    
    fieldsets = (
        ('Lab Test Information', {
            'fields': ('lab_test',)
        }),
        ('Biomarker Information', {
            'fields': ('biomarker', 'biomarker_name', 'test_code')
        }),
        ('Test Result', {
            'fields': ('value', 'value_string', 'unit', 'method')
        }),
        ('Reference Range (from lab)', {
            'fields': ('ref_min', 'ref_max', 'ref_label'),
            'classes': ('collapse',)
        }),
        ('Analysis Results', {
            'fields': ('fuzzy_score', 'status', 'lifestyle_feedback')
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    def get_queryset(self, request):
        """Optimize queryset with select_related"""
        qs = super().get_queryset(request)
        return qs.select_related('lab_test', 'lab_test__user', 'lab_test__user__profile', 'biomarker')


admin.site.register(LabTestResult, LabTestResultAdmin)


class AIChatMessageAdmin(admin.ModelAdmin):
    list_display = ('id', 'user', 'user_message_preview', 'ai_response_preview', 'created_at', 'conversation_id')
    list_filter = ('created_at', 'conversation_id')
    search_fields = ('user__phone_number', 'user__profile__first_name', 'user__profile__last_name', 
                     'user_message', 'ai_response', 'conversation_id')
    readonly_fields = ('created_at',)
    date_hierarchy = 'created_at'
    ordering = ('-created_at',)
    list_per_page = 25
    
    fieldsets = (
        ('User Information', {
            'fields': ('user',)
        }),
        ('Message Content', {
            'fields': ('user_message', 'ai_response')
        }),
        ('Conversation', {
            'fields': ('conversation_id',),
            'classes': ('collapse',)
        }),
        ('Timestamps', {
            'fields': ('created_at',),
            'classes': ('collapse',)
        }),
    )
    
    def user_message_preview(self, obj):
        """Display truncated user message preview"""
        if len(obj.user_message) > 60:
            return obj.user_message[:60] + "..."
        return obj.user_message
    user_message_preview.short_description = 'User Message'
    
    def ai_response_preview(self, obj):
        """Display truncated AI response preview"""
        if len(obj.ai_response) > 60:
            return obj.ai_response[:60] + "..."
        return obj.ai_response
    ai_response_preview.short_description = 'AI Response'
    
    def get_queryset(self, request):
        """Optimize queryset with select_related"""
        qs = super().get_queryset(request)
        return qs.select_related('user', 'user__profile')


admin.site.register(AIChatMessage, AIChatMessageAdmin)


class FoodRecommendationsCacheAdmin(admin.ModelAdmin):
    list_display = ('id', 'user', 'total_foods', 'created_at', 'updated_at')
    list_filter = ('created_at', 'updated_at')
    search_fields = ('user__phone_number', 'user__profile__first_name', 'user__profile__last_name', 'diseases_string')
    readonly_fields = ('created_at', 'updated_at')
    date_hierarchy = 'created_at'
    ordering = ('-updated_at',)
    list_per_page = 25
    
    fieldsets = (
        ('User Information', {
            'fields': ('user',)
        }),
        ('Recommendations Data', {
            'fields': ('recommendations', 'total_foods', 'diseases_string', 'lab_results')
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at')
        }),
    )
    
    def get_queryset(self, request):
        """Optimize queryset with select_related"""
        qs = super().get_queryset(request)
        return qs.select_related('user', 'user__profile')


admin.site.register(FoodRecommendationsCache, FoodRecommendationsCacheAdmin)
